package risiko.common.interfaces;

import java.net.MalformedURLException;
import java.rmi.NotBoundException;
import java.rmi.Remote;
import java.rmi.RemoteException;
import java.util.Vector;

import risiko.common.exceptions.KeineNachbarnException;
import risiko.common.exceptions.LandBeteiligtException;
import risiko.common.exceptions.LandNichtImBesitzException;
import risiko.common.exceptions.NichtBenachbartException;
import risiko.common.exceptions.NichtGenugEinheitenException;
import risiko.common.exceptions.NichtGenugWuerfelException;
import risiko.common.exceptions.VerteidigerUebersteigtAngreiferException;

/**
 * Remote-Interface des Servers
 * 
 * @author dgrosche
 * @version 2009-07-01
 *
 */
public interface SpielMgr extends Remote {

	/**
	 * Fgt neuen Spieler hinzu.
	 * 
	 * @param name Spielername
	 */
	public abstract int addSpieler(String name) throws RemoteException;

	/**
	 * Die Lnder werden gleichmig unter die Spieler aufgeteilt.
	 */
	public abstract void verteileLaender() throws RemoteException;

	/**
	 * Die Missionen werden erstellt und auf die Spieler aufgeteilt.
	 */
	public abstract void verteileMissionen() throws RemoteException;

	/**
	 * Missioenen werden ersetzt durch Welterobermissionen
	 */
	public abstract void weltErobern() throws RemoteException;

	/**
	 * Die Startreihenfolge wird ausgewrfelt.
	 * 
	 * @return Vektor mit Wrfelzahlen
	 */
	public abstract Vector<Integer> reihenfolge() throws RemoteException;

	/**
	 * Setzt entsprechend der Spielphase den Modus weiter.
	 * <br>
	 *   <br><b> Phase 0:   </b> Hauptmen					
	 *   <br><b> Phase 1:   </b> Angriffsmen				
	 *   <br><b> Phase 2:   </b> Verschiebenmen			
	 *   <br><b> Phase 3:   </b> Lnder drucken			
	 *   <br><b> Phase 4:   </b> Zug beenden				
	 *   <br><b> Phase 5:   </b> Karten einlsen			
	 *   <br><b> Phase 6:   </b> Spiel speichern			
	 *   <br><b> Phase 7:   </b> Spiel laden				
	 *   <br><b> Phase 50:  </b> Verteidigen				
	 *   <br><b> Phase 60:  </b> Verstrkung setzen		
	 *   <br><b> Phase 70:  </b> Einheiten nachrcken		
	 *   <br><b> Phase 100: </b> Spiel beenden				
	 * 
	 */
	public abstract void weiter() throws RemoteException;

	/**
	 * Setzt die Spielphase auf einen bestimmten Wert
	 * <br>
	 *   <br><b> Phase 0:   </b> Hauptmen					
	 *   <br><b> Phase 1:   </b> Angriffsmen				
	 *   <br><b> Phase 2:   </b> Verschiebenmen			
	 *   <br><b> Phase 3:   </b> Lnder drucken			
	 *   <br><b> Phase 4:   </b> Zug beenden				
	 *   <br><b> Phase 5:   </b> Karten einlsen			
	 *   <br><b> Phase 6:   </b> Spiel speichern			
	 *   <br><b> Phase 7:   </b> Spiel laden				
	 *   <br><b> Phase 50:  </b> Verteidigen				
	 *   <br><b> Phase 60:  </b> Verstrkung setzen		
	 *   <br><b> Phase 70:  </b> Einheiten nachrcken		
	 *   <br><b> Phase 100: </b> Spiel beenden					
	 * 
	 * @param wert Wert auf den Phase gesetzt werden soll
	 */
	public abstract void setPhase(int wert) throws RemoteException;

	/**
	 * Gibt die Spielphase zurck
	 * <br>
	 *   <br><b> Phase 0:   </b> Hauptmen					
	 *   <br><b> Phase 1:   </b> Angriffsmen				
	 *   <br><b> Phase 2:   </b> Verschiebenmen			
	 *   <br><b> Phase 3:   </b> Lnder drucken			
	 *   <br><b> Phase 4:   </b> Zug beenden				
	 *   <br><b> Phase 5:   </b> Karten einlsen			
	 *   <br><b> Phase 6:   </b> Spiel speichern			
	 *   <br><b> Phase 7:   </b> Spiel laden				
	 *   <br><b> Phase 50:  </b> Verteidigen				
	 *   <br><b> Phase 60:  </b> Verstrkung setzen		
	 *   <br><b> Phase 70:  </b> Einheiten nachrcken		
	 *   <br><b> Phase 100: </b> Spiel beenden			
	 * 
	 * @return aktuelle Phase
	 */
	public abstract int getPhase() throws RemoteException;

	/**
	 * Startet einen neuen Angriff von a nach b mit x Angreifern <i>(Spielphase: 1)</i>
	 * 
	 * @param a angreifendes Land
	 * @param b verteidigendes Land
	 * @param x Anzahl der Einheiten fr den Angriff
	 * @throws LandNichtImBesitzException , wenn a nicht im Besitz des aktiven Spielers
	 * @throws NichtBenachbartException , wenn a und b nicht benachtbart
	 * @throws NichtGenugEinheitenException , wenn a nur 1 Einheit auf dem Land hat 
	 * @throws NichtGenugWuerfelException 
	 * @throws NichtGenugWuerfelException , wenn x grer als 3.
	 */
	public abstract void starteAngriff(Land a, Land b, int x)
			throws LandNichtImBesitzException, NichtBenachbartException,
			NichtGenugEinheitenException, NichtGenugWuerfelException, RemoteException;

	/**
	 * Verschiebt x Einheiten von Land a nach Land b <i>(Spielphase: 2)</i>
	 * 
	 * @param a Ursprungsland
	 * @param b Zielland
	 * @param x Anzahl der Einheiten
	 * @throws NichtGenugEinheitenException , wenn nicht genug Einheiten im Land a verbleiben
	 * @throws LandBeteiligtException , wenn eines der Lnder bereits am Spielzug beteiligt war
	 * @throws LandNichtImBesitzException , wenn eines der Lnder nicht im Besitz des aktiven Spielers 
	 * @throws NichtBenachbartException , wenn a und b nicht benachbart sind
	 */
	public abstract void verschiebeEinheiten(Land a, Land b, int x)
			throws NichtGenugEinheitenException, LandBeteiligtException,
			LandNichtImBesitzException, NichtBenachbartException, RemoteException;

	/**
	 * <br>Setzt nchsten Spieler in der Reihe aktiv.
	 * <br>Fgt Karte hinzu falls im Zug Lnder erobert wurden.
	 * <br><i>(Spielphase: 4)</i>
	 */
	public abstract void beendeZug() throws RemoteException;

	/**
	 * Lst einen Kartensatz ein <i>(Spielphase: 5)</i>
	 * 
	 * @param kartensatz einzulsender Kartensatz
	 */
	public abstract void kartenEinloesen(int[] kartensatz) throws RemoteException;

	/**
	 * Speichert den Spielstand <i>(Spielphase: 6)</i>
	 * 
	 * @return Fehlermeldung oder Besttigung
	 */
	public abstract String speichern() throws RemoteException;

	/**
	 * Ldt letzten Spielstand <i>(Spielphase: 7)</i>
	 * 
	 * @return Fehlermeldung oder Besttigung
	 */
	public abstract String laden() throws RemoteException;

	/**
	 * Legt Verteidigung fest und fhrt aktuellen Kampf durch. <i>(Spielphase: 50)</i>
	 * 
	 * @param y Anzahl der Einheiten zur Verteidigung
	 * @throws NichtGenugWuerfelException , wenn mehr als 3 Angreifer oder 2 Verteidiger
	 * @throws VerteidigerUebersteigtAngreiferException , wenn mehr Verteidiger als Angreifer
	 * @throws NichtGenugEinheitenException , wenn fr die Anzahl der Einheiten im Kampf (sowohl Angreifer als auch Verteidiger) nicht gengend Einheiten vorhanden sind.
	 */
	public abstract void verteidige(int y) throws NichtGenugWuerfelException,
			VerteidigerUebersteigtAngreiferException,
			NichtGenugEinheitenException, RemoteException;

	/**
	 * Die zustlichen Einheiten fr aktuellen Spieler werden ermittelt. <i>(Spielphase: 60)</i> 
	 * 
	 * @return Anzahl der zustzlichen Einheiten
	 */
	public abstract int getVerstaerkung() throws RemoteException;

	/**
	 * Prft ob einlsbare Karten vorhanden sind <i>(Spielphase: 5)</i>
	 * 
	 * @return true, wenn min. ein Satz einlsbarer Karten
	 */
	public abstract boolean pruefKarten() throws RemoteException;

	/**
	 * @return the einloesbareKarten
	 */
	public abstract Vector<int[]> getEinloesbareKarten() throws RemoteException;

	/**
	 * Prft ob aktiver Spieler gewonnen hat.
	 * 
	 * @return true, wenn aktiver Spieler seine Mission erfllt hat
	 */
	public abstract boolean isGewonnen() throws RemoteException;
	
	/**
	 * Gibt Verteidigerland zurck
	 * 
	 * @return verteidigendes Land
	 */
	public abstract Land getVerteidiger() throws RemoteException;
	
	/**
	 * Gibt Angreiferland zurck
	 * 
	 * @return  angreifendes Land
	 */
	public abstract Land getAngreifer() throws RemoteException;
	
	/**
	 * Wrfelzahlen des Angreifers
	 * 
	 * @return Vektor mit Wrfelzahlen
	 */
	public abstract Vector<Integer> getWuerfelAngreifer() throws RemoteException;
	
	/**
	 * Wrfelzahlen des Verteidigers
	 * 
	 * @return Vektor mit Wrfelzahlen
	 */
	public abstract Vector<Integer> getWuerfelVerteidiger() throws RemoteException;
	
	/**
	 * Gibt Anzahl angreifender Einheiten zurck
	 * 
	 * @return angreifende Einheiten
	 */
	public abstract int getEinheitenAngriff() throws RemoteException;

	/**
	 * Gibt aktiven Spieler zurck
	 * 
	 * @return aktiven Spieler
	 */
	public abstract Spieler getAktiverSpieler() throws RemoteException;

	/**
	 * Gibt alle Spieler zurck
	 * 
	 * @return Vektor mit allen Spielern
	 */
	public abstract Vector<Spieler> getAllSpieler() throws RemoteException;

	/**
	 * Gibt komplette Lnderliste zurck
	 * 
	 * @return Vektor mit allen Lndern
	 */
	public abstract Vector<Land> getAllLaender() throws RemoteException;

	/**
	 * Gibt alle Lnder eines Spielers zurck
	 * 
	 * @param spieler Spieler
	 * @return Vektor mit Lndern des Spielers
	 */
	public abstract Vector<Land> getLaender(Spieler spieler) throws RemoteException;

	/**
	 * Gibt alle Nachbarlnder eines Landes zurck
	 * 
	 * @param a Land dessen Nachbarn gesucht sind
	 * @return Vektor mit allen Nachbarn des Landes
	 */
	public abstract Vector<Land> getNachbarn(Land a) throws RemoteException;

	/**
	 * Gibt alle eigenen Nachbarn zurck
	 * 
	 * @param a Land dessen eigene Nachbarn gesucht sind
	 * @return Vektor mit allen benachbarten Lndern mit gleichen Besitzer von Land a
	 * @throws KeineNachbarnException , wenn keine eigenen Nachbarn
	 */
	public abstract Vector<Land> getEigeneNachbarn(Land a)
			throws KeineNachbarnException, RemoteException;

	/**
	 * Gibt alle fremden Nacharn zurck
	 * 
	 * @param a Land dessen fremde Nachbarn gesucht sind
	 * @throws KeineNachbarnException , wenn keine fremden Nachbarn vorhanden
	 * @return Vektor mit allen benachbarten Lndern mit anderen Besitzer als a
	 */
	public abstract Vector<Land> getFremdeNachbarn(Land a)
			throws KeineNachbarnException, RemoteException;

	public abstract boolean isNachbar(Land a, Land b) throws RemoteException;

	/**
	 * Fgt x Einheiten zu Land l hinzu.
	 * 
	 * @param l Land
	 * @param x Anzahl Einheiten
	 */
	public abstract void setzeEinheiten(Land l, int x) throws RemoteException;
	
	/**
	 * @param anzSpieler the anzSpieler to set
	 */
	public abstract void setAnzSpieler(int anzSpieler) throws RemoteException; 
	
	/**
	 * @return the anzSpieler
	 */
	public abstract int getAnzSpieler() throws RemoteException;
	
	/**
	 * @param spielstart the spielstart to set
	 */
	public abstract void setSpielstart(boolean spielstart) throws RemoteException;
	
	/**
	 * @return the spielstart
	 */
	public abstract boolean isSpielstart() throws RemoteException;
	
	/**
	 * Fgt einen Client als Observer hinzu 
	 * @throws NotBoundException 
	 * @throws MalformedURLException 
	 */
	public void addClient(String servicename) throws RemoteException, MalformedURLException, NotBoundException;
	
	/**
	 * Benachrichtigt alle Clients 
	 */
	public abstract void notifyClients(Object arg) throws RemoteException;
}