package risiko.client.ui.cui;

import java.awt.Button;
import java.awt.Color;
import java.awt.Frame;
import java.awt.TextArea;
import java.rmi.Naming;
import java.rmi.RMISecurityManager;
import java.rmi.RemoteException;
import java.util.Observable;
import java.util.Observer;
import java.util.Vector;

import risiko.client.ui.Client;
import risiko.common.exceptions.KeineNachbarnException;
import risiko.common.exceptions.LandBeteiligtException;
import risiko.common.exceptions.LandNichtImBesitzException;
import risiko.common.exceptions.NichtBenachbartException;
import risiko.common.exceptions.NichtGenugEinheitenException;
import risiko.common.exceptions.NichtGenugWuerfelException;
import risiko.common.exceptions.VerteidigerUebersteigtAngreiferException;
import risiko.common.interfaces.Land;
import risiko.common.interfaces.SpielMgr;

/**
 * Kommandezeilen-basierte Benutzeroberflche als eigenstndiger Thread
 * 
 * @author dgrosche
 * @version 2009-05-31
 *
 */
public class CUI extends Thread implements Observer {

	/**
	 * Hauptklasse (Ablaufplan des Threads)
	 */
	public synchronized void run() {
		try {
			println("Herzlich Willkommen beim Konsolen-Risiko");
			println("****************************************");
			println("");
			println("[1] Neues Spiel starten");
			println("[2] Spiel laden");
			println("");
			int auswahl = 0;
			print("Auswahl: ");
			while(!((auswahl>0)&&(auswahl<3))) {
				try {
					auswahl = Integer.parseInt(einlesen());
				} catch(NumberFormatException e1) {
					auswahl = 0;
					printError(e1);
				}
			}
			println("");
			if(auswahl==1) neuesSpiel();
			if(auswahl==2) laden();
			while(risiko.getPhase()!=100) {
//				risiko.notifyClients(null);
				this.spielerMenu();
			}
			if(this.risiko.isGewonnen()) {
				println("****************************************");
				println("****************************************");
				println("****************************************");
				println("");
				println("Herzlichen Glckwunsch, "+this.risiko.getAktiverSpieler().getName()+"!");
				println("");
				println("Du hast gewonnen!");
				println("");
				this.printLaender();
			}
		} catch(RemoteException e) {
			printError(e);
		}

	}

	/**
	 * Gelesene Eingabe, wird von Konsole berschrieben
	 */
	public String in;

	/**
	 * Ausgabe-Feld
	 */
	public TextArea out;

	/**
	 * Senden-Button bermittelt Eingabe von Konsole
	 */
	public Button senden;

	/**
	 * Gibt an ob der Programmablauf fortgefahren werden kann
	 */
	public boolean weiter;

	private SpielMgr risiko;
	private Client clientObserver;
	private int client;

	/**
	 * Konstruktor mit Hauptverwaltungsklasse 
	 * 
	 * @param out Ausgabekonsole
	 * @param senden Button der Eingabe bermittelt
	 */
	public CUI(TextArea out, Button senden, String ip) {
		this.in = "";
		if (System.getSecurityManager() == null) {
            System.setSecurityManager(new RMISecurityManager());
        }
		try {
			this.risiko = (SpielMgr) Naming.lookup("rmi://"+ip+"/RisikoService");
			clientObserver = new Client(this,ip);
			risiko.addClient(clientObserver.getServicename());
		} catch(Exception e) {
			e.printStackTrace();
		}
		this.out = out;
		this.senden = senden;
	}

	/**
	 * Unterbricht den Thread und wartet auf eine Eingabe
	 * 
	 * @return Eingabestring
	 */
	public String einlesen() {
		senden.setEnabled(true);

		try {
			this.wait();
		} catch(InterruptedException e) {
		}
		if(weiter) {
			this.println(in);
			weiter = false;
			senden.setEnabled(false);
		}
		return in;
	}

	/**
	 * Fgt Konsole Text und Zeilenumbruch hinzu 
	 * 
	 * @param text auszugebender Text
	 */
	public void println(String text) {
		out.setForeground(Color.BLACK);
		out.append(text+"\n");
		out.repaint();
	}

	/**
	 * Fgt Konsole Text ohne Zeilenumbruch hinzu
	 * 
	 * @param text auszugebender Text
	 */
	public void print(String text) {
		out.setForeground(Color.BLACK);
		out.append(text);
		out.repaint();
	}

	/**
	 * Fgt Konsole Fehlermeldung hinzu und ndert Schriftfarbe auf Rot
	 * 
	 * @param e auszugebender Fehler
	 */
	public void printError(Exception e) {
		out.setForeground(Color.RED);
		out.append(e.toString()+"\n");
		out.repaint();
	}

	/*
	 * Neues Spiel erstellen
	 */

	/**
	 * Initialisiert das Spiel, d.h.
	 * <ul>
	 * <li>		Spieler werden registriert			</li>
	 * <li>		Lnder werden verteilt				</li>
	 * <li>		Startreihenfolge wird festgelegt	</li>
	 * <li>		Anfangseinheiten werden gesetzt		</li>
	 * </ul>
	 * @throws RemoteException 
	 */
	public void neuesSpiel() throws RemoteException {
		//Anzahl der Spieler wird bestimmt. Die Zahl muss zwischen 1 und 6 liegen
		if(risiko.getAllSpieler().size()==0) {
			print("Anzahl der Spieler: ");
			int anzSpieler = 0;
			while(!(anzSpieler>2 && anzSpieler<7)) {
				try {
					anzSpieler = Integer.parseInt(einlesen());
				} catch (NumberFormatException e) {
					anzSpieler = 0;
					printError(e);
				}
			}
			risiko.setAnzSpieler(anzSpieler);
		}

		//Spieler werden hinzugefgt
		println("");
		println((risiko.getAllSpieler().size()+1)+". Spieler");
		println("++++++++++");
		this.addSpieler();
		println("****************************************");
		
		while(risiko.getAllSpieler().size()!=risiko.getAnzSpieler());

		if(client==0) {
			//Spieler erhalten Lnder und Missioen
			risiko.verteileLaender();
			risiko.verteileMissionen();

			//Missionen oder Welt erobern
			println("[1] Missionen ausfhren");
			println("[2] Welt erobern");
			println("");
			int auswahl = 0;
			print("Auswahl: ");
			while(!((auswahl>0)&&(auswahl<3))) {
				try {
					auswahl = Integer.parseInt(einlesen());
				} catch(NumberFormatException e1) {
					auswahl = 0;
					printError(e1);
				}
			}
			if(auswahl==2) risiko.weltErobern();
			println("****************************************");

			//Die Startreihenfolge wird ausgewrfelt
			Vector<Integer> zahlen = risiko.reihenfolge();
			String startreihenfolge = "Fr die Startreihenfolge wurden folgende Zahlen gewrfelt:";
			for(int i=0; i<zahlen.size(); i++) {
				startreihenfolge += "   "+risiko.getAllSpieler().elementAt(i).getName()+": "+zahlen.elementAt(i);
				risiko.beendeZug();
			}
			addLogEintrag(startreihenfolge);
			risiko.setSpielstart(true);
		}
		
		while(!risiko.isSpielstart());

		//Der erste Spieler beginnt mit dem Setzen von zustzlichen Einheiten
		risiko.setPhase(60);		
	}

	/**
	 * Fragt die Spielernamen ab und fgt sie hinzu.
	 * @throws RemoteException 
	 */
	public void addSpieler() throws RemoteException {
		print("Name des Spielers: ");
		String name = "";
		while (name.equals("")) {
			try {
				name = einlesen();
			} catch (Exception e) {
				name = "";
				printError(e);
			}
		}
		client = risiko.addSpieler(name)-1;
		Frame frame = (Frame) senden.getParent().getParent();
		frame.setTitle(name);
	}

	/*
	 * Menfhrung
	 */

	/**
	 * Gibt das Spielphasen entsprechende Men aus:
	 * <br>
	 *   <br><b> Phase 0:   </b> Hauptmen					
	 *   <br><b> Phase 1:   </b> Angriffsmen				
	 *   <br><b> Phase 2:   </b> Verschiebenmen			
	 *   <br><b> Phase 3:   </b> Lnder drucken			
	 *   <br><b> Phase 4:   </b> Zug beenden				
	 *   <br><b> Phase 5:   </b> Karten einlsen			
	 *   <br><b> Phase 6:   </b> Spiel speichern			
	 *   <br><b> Phase 7:   </b> Spiel laden				
	 *   <br><b> Phase 50:  </b> Verteidigen				
	 *   <br><b> Phase 60:  </b> Verstrkung setzen		
	 *   <br><b> Phase 70:  </b> Einheiten nachrcken		
	 *   <br><b> Phase 100: </b> Spiel beenden			
	 * @throws RemoteException 
	 *   
	 */
	public void spielerMenu() throws RemoteException {
		if(client==risiko.getAktiverSpieler().getNummer()) {
			switch (risiko.getPhase()) {
			case 0: hauptmenu(); break;
			case 1: neuerAngriff(); break;
			case 2: verschieben(); break;
			case 3: risiko.weiter(); break;
			case 4: printLaender(); break; 
			case 5: kartenEinloesen(); break;
			case 6: speichern(); break;
//			case 7: laden(); break;
//			case 50: verteidige(); break;
			case 60: verstaerkung(); break;
			case 70: nachruecken(); break;
			}
		} else if(risiko.getPhase()==50) {
			if(client==risiko.getVerteidiger().getBesitzer().getNummer()) {
				verteidige();
			}
		} else {
			warten();
		}
	}

	/**
	 * Gibt das Hauptmen eines Spielers aus <i>(Spielphase: 0)</i>
	 * @throws RemoteException 
	 */
	public void hauptmenu() throws RemoteException {
		println("");
		println("****************************************");
		println(risiko.getAktiverSpieler().getName()+", du hast folgende Mglichkeiten:");
		println("");
		println("MISSION: "+risiko.getAktiverSpieler().getMission().getMessage());
		println("");
		println("   [1] Angriff starten");
		println("   [2] Einheiten verschieben");
		println("   [3] Zug beenden");
		println("");
		println("   [4] Eigene Lnder anzeigen");
		println("   [5] Karten eintauschen");
		println("");
		if(client==0) println("   [6] Spiel speichern");
//		println("   [7] Spiel laden");
//		println("   [8] Spiel beenden");
		println("");
		print("Auswahl: ");
		int auswahl = 0;
		int max = (client==0) ? 6 : 5;
		while(!(auswahl>0 && auswahl<=max)) {
			try {
				auswahl = Integer.parseInt(einlesen());
			} catch (NumberFormatException e) {
				auswahl = 0;
				printError(e);
			}
		}
		println("****************************************");
		println("");

		//Die gewhlte Zahl wird direkt als nchste Phase weitergegeben
//		if(auswahl==8) auswahl = 100; //... nur bei Spiel beenden nicht
		risiko.setPhase(auswahl);
	}

	/**
	 * Startet neuen Kampf <i>(Spielphase: 1)</i>
	 * @throws RemoteException 
	 */
	public void neuerAngriff() throws RemoteException {
		println("Von wo willst du angreifen?");
		this.printLaender();
		print("Nummer eingeben: ");
		int landNr = 0;
		while(!(landNr>0 && landNr<=risiko.getLaender(risiko.getAktiverSpieler()).size())) {
			try {
				landNr = Integer.parseInt(einlesen());
			} catch (NumberFormatException e) {
				landNr = 0;
				printError(e);
			} 
		}
		landNr--;
		Land angriffsland = risiko.getLaender(risiko.getAktiverSpieler()).elementAt(landNr);
		try {
			Vector<Land> fremdeNachbarn = risiko.getFremdeNachbarn(angriffsland);
			println("");
			println("Welches Land soll angegriffen werden?");
			int i = 1;
			for(Land l : fremdeNachbarn) {
				println("   "+(i)+")"+l.getString()+" ["+l.getBesitzer().getName()+"]");
				i++;
			}
			print("Nummer eingeben: ");
			landNr = 0;
			while(!(landNr>0 && landNr<=fremdeNachbarn.size())) {
				try {
					landNr = Integer.parseInt(einlesen());
				} catch (NumberFormatException e) {
					landNr = 0;
					printError(e);
				} 
			}
			landNr--;
			Land verteidigungsland = fremdeNachbarn.elementAt(landNr);

			print("Mit wievielen Einheiten soll angegriffen werden? ");
			int angreifer = 0;
			while(!(angreifer>0)) {
				try {
					angreifer = Integer.parseInt(einlesen());
				} catch (NumberFormatException e) {
					angreifer = 0;
					printError(e);
				}
			}

			//Sind die Eingaben formal richtig, wird ein Angriff gestartet
			risiko.starteAngriff(angriffsland, verteidigungsland, angreifer);
			addLogEintrag("Angriff aus "+angriffsland.getName()
					+" nach "+verteidigungsland.getName()+" mit "+angreifer+" Einheiten");
			risiko.weiter();
		} catch (LandNichtImBesitzException e) {
			printError(e);
		} catch (KeineNachbarnException e) {
			printError(e);
			risiko.setPhase(0); //Es kann nicht angegriffen werden, also Hauptmen 
		} catch (NichtBenachbartException e) {
			printError(e);
			risiko.setPhase(0); //Es kann nicht angegriffen werden, also Hauptmen
		} catch (NichtGenugEinheitenException e) {
			printError(e);
			risiko.setPhase(0); //Es kann nicht angegriffen werden, also Hauptmen
		} catch (NichtGenugWuerfelException e) {
			printError(e);
		}
	}

	/**
	 * Verschiebt Einheiten <i>(Spielphase: 2)</i>
	 * @throws RemoteException 
	 */
	public void verschieben() throws RemoteException {
		println("Verschieben aus Land: ");
		this.printLaender();
		print("Nummer eingeben: ");
		int landNr = 0;
		while(!(landNr>0 && landNr<=risiko.getLaender(risiko.getAktiverSpieler()).size())) {
			try {
				landNr = Integer.parseInt(einlesen());
			} catch (NumberFormatException e) {
				landNr = 0;
				printError(e);
			}
		}
		landNr--;
		Land ursprungsland = risiko.getLaender(risiko.getAktiverSpieler()).elementAt(landNr);
		println("");
		try {
			int i=1;
			for(Land l : risiko.getEigeneNachbarn(ursprungsland)) {
				println("   "+(i)+")"+l.getString());
				i++;
			}
			print("Nummer eingeben: ");
			landNr = 0;
			while(!(landNr>0 && landNr<=risiko.getEigeneNachbarn(ursprungsland).size())) {
				try {
					landNr = Integer.parseInt(einlesen());
				} catch (NumberFormatException e) {
					landNr = 0;
					printError(e);
				}
			}
			landNr--;
			Land zielland = risiko.getEigeneNachbarn(ursprungsland).elementAt(landNr);
			print("Wie viele Einheiten sollen verschoben werden? ");
			int anzEinheiten = -1; 
			while(!(anzEinheiten>=0 && anzEinheiten<ursprungsland.getAnzEinheiten())) {
				try {
					anzEinheiten = Integer.parseInt(einlesen());
				} catch (NumberFormatException e) {
					anzEinheiten = 0;
					printError(e);
				}
			}

			//Sind die Eingaben formal richtig, werden die Einheiten verschoben
			risiko.verschiebeEinheiten(ursprungsland, zielland, anzEinheiten);
			addLogEintrag("Verschieben von "+anzEinheiten+" Einheiten aus "
					+ursprungsland.getName()+" nach "+zielland.getName());

			//Das Verschieben wird solange wiederholt, wie man will
			print("Nochmal verschieben? (j/n): ");
			char nochmal ='0';
			while(!((nochmal=='j')||(nochmal=='J')||(nochmal=='n')||(nochmal=='N'))) {
				try {
					nochmal = einlesen().charAt(0);
				} catch (Exception e) {
					nochmal = '0';
					printError(e);
				}
			}

			//Wird nein ausgwhlt, geht's zur nchsten Phase weiter
			if((nochmal=='n')||(nochmal=='N')) risiko.weiter();

		} catch (NichtGenugEinheitenException e) {
			printError(e);
		} catch (KeineNachbarnException e) {
			printError(e);
		} catch (LandBeteiligtException e) {			
			printError(e);
		} catch (LandNichtImBesitzException e) {
			printError(e);
		} catch (NichtBenachbartException e) {
			printError(e);
		}		
	}

	/**
	 * Gibt alle Lnder des aktiven Spieler aus <i>(Spielphase: 3)</i>
	 * @throws RemoteException 
	 */
	public void printLaender() throws RemoteException {
		println(risiko.getAktiverSpieler().getName()+", deine Lnder sind:");
		for(int i=0; i<risiko.getLaender(risiko.getAktiverSpieler()).size(); i++) {
			println("   "+(i+1)+") "+risiko.getLaender(risiko.getAktiverSpieler()).elementAt(i).getString());
		}
		println("");

		//Wenn die Methode als Spielphase aufgerufen wird, muss die Phase anschlieend verndert werden.
		if(risiko.getPhase()==4) risiko.weiter();
	}

	/**
	 * <br>Gibt Karten zurck. 
	 * <br>Prft ob einlsbare Karten vorhanden sind.
	 * <br>Lsst Karten einlsen.
	 * <br><i>(Spielphase: 5)</i> 
	 * @throws RemoteException 
	 */
	public void kartenEinloesen() throws RemoteException {
		boolean pflicht = (risiko.getAktiverSpieler().getKarten().size()>4) ? true : false;
		println(risiko.getAktiverSpieler().getName()+", du hast folgende Karten:");
		for(int i : risiko.getAktiverSpieler().getKarten()) {
			//Die Karten sind eigentlich nur Zahlen, entsprechend dieser Liste
			switch(i) {
			case 0: println("   Reiter"); break;
			case 1: println("   Kanonen"); break;
			case 2: println("   Soldaten"); break;
			case 3: println("   Joker"); break;
			default: print(""); break;
			}
		}
		println("");
		if(!risiko.pruefKarten()) {
			println("Du kannst keine Karten eintauschen!");
			risiko.weiter();
		} else {
			println("Du kannst Karten eintauschen!");
			Vector<int[]> karten = risiko.getEinloesbareKarten(); 
			if(!pflicht) {
				println("   0) keine Karten eintauschen!");
			}
			for(int i=0; i<karten.size(); i++) {
				print("   "+(i+1)+") ");
				for(int j=0; j<3; j++) {
					switch(karten.elementAt(i)[j]) {
					case 0: print("Reiter, "); break;
					case 1: print("Kanonen, "); break;
					case 2: print("Soldaten, "); break;
					case 3: print("Joker, "); break;
					}
				}
				println("");
			}
			println("");
			print("Nummer eingeben: ");
			int min = pflicht ? 0 : -1;
			int kartSatzNr = min;
			while(!(kartSatzNr>min && kartSatzNr<=karten.size())) {
				try {
					kartSatzNr = Integer.parseInt(einlesen());
				} catch (NumberFormatException e) {
					kartSatzNr = min;
					printError(e);
				}
			}
			if(kartSatzNr>0) {
				kartSatzNr--;
				risiko.kartenEinloesen(karten.elementAt(kartSatzNr));
			}
			if(pflicht) {
				risiko.setPhase(60);
			} else {
				risiko.weiter();
			}
		}		
	}

	/**
	 * Speichert einen neuen Spielstand <i>(Spielphase: 6)</i>
	 * @throws RemoteException 
	 */
	public void speichern() throws RemoteException {
		println("****************************************");
		println(risiko.speichern());
		println("****************************************");
		risiko.weiter();
	}

	/**
	 * Ldt den letzten vorhandenen Spielstand <i>(Spielphase: 7)</i>
	 * @throws RemoteException 
	 */
	public void laden() throws RemoteException {
		addLogEintrag(risiko.laden());
		println("****************************************");
		println("Whle dein Spielerprofil:");
		println("");
		for(int i=0; i<risiko.getAllSpieler().size(); i++) {
			println((i+1)+") "+risiko.getAllSpieler().elementAt(i).getName());
		}
		println("");
		int auswahl = 0;
		print("Auswahl: ");
		while(!((auswahl>0)&&(auswahl<=risiko.getAllSpieler().size()))) {
			try {
				auswahl = Integer.parseInt(einlesen());
			} catch(NumberFormatException e1) {
				auswahl = 0;
				printError(e1);
			}
		}
		println("");
		client = auswahl-1;
		Frame frame = (Frame) senden.getParent().getParent();
		frame.setTitle(risiko.getAllSpieler().elementAt(client).getName());
		println("****************************************");
		risiko.weiter();
	}

	/**
	 * Fhrt Verteidigung aus <i>(Spielphase: 50)</i>
	 * @throws RemoteException 
	 */
	public void verteidige() throws RemoteException {
		println("****************************************");
		Land verteidigungsland = risiko.getVerteidiger();
		println(verteidigungsland.getBesitzer().getName()+", du wirst angegriffen!");
		println("   Angreifer: "+risiko.getAngreifer().getBesitzer().getName());
		println("   Angriff aus: "+risiko.getAngreifer().getString());
		println("   Angriff nach: "+risiko.getVerteidiger().getName());
		println("");
		print("Whle deine Verteidigungsstrke: ");
		int verteidiger = 0;

		while(!(verteidiger>0)) {
			try {
				verteidiger = Integer.parseInt(einlesen());
			} catch (NumberFormatException e) {
				verteidiger = 0;
				printError(e);
			}
		}
		println("****************************************");
		String verteidigername = verteidigungsland.getBesitzer().getName();
		Land angriffsland = risiko.getAngreifer();
		try {
			//Sind die Eingaben korrekt, wird verteidigt und die Zahlen werden ausgedruckt
			risiko.verteidige(verteidiger);
			String wuerfel = angriffsland.getBesitzer().getName()+", deine Wrfelzahlen:\n";
			for(int zahl : risiko.getWuerfelAngreifer()) {
				wuerfel += "   "+zahl+"\n";
			}
			wuerfel+="\n";
			wuerfel+=verteidigername+", deine Wrfelzahlen:\n";
			for(int zahl : risiko.getWuerfelVerteidiger()) {
				wuerfel+="   "+zahl+"\n";
			}
			addLogEintrag(wuerfel);
			risiko.weiter();
		} catch (NichtGenugWuerfelException e) {
			printError(e);
			//Wenn der Fehler vom Angreifer ausgeht, gehts zum Hauptmen
			if(e.behandle()) risiko.weiter();
		} catch (VerteidigerUebersteigtAngreiferException e) {
			printError(e);
		} catch (NichtGenugEinheitenException e) {
			printError(e);
			//Wenn der Fehler vom Angreifer ausgeht, gehts zum Hauptmen
			if(e.behandle()) risiko.weiter();
		}
	}

	/**
	 * Lsst einen Spieler Zusatzeinheiten verteilen <i>(Spielphase: 60)</i>
	 * @throws RemoteException 
	 */
	public void verstaerkung() throws RemoteException {
		println("****************************************");
		println("");
		println("MISSION: "+risiko.getAktiverSpieler().getMission().getMessage());
		println("");
		println(risiko.getAktiverSpieler().getName()+", du kannst "+risiko.getVerstaerkung()+" zustzliche Einheiten verteilen:");
		this.printLaender();
		for(int i=0; i<risiko.getVerstaerkung(); i++) {
			print("Einheiten hinzufgen zu Land Nummer: ");
			int landNr = 0;
			while(!(landNr>0 && landNr<=risiko.getLaender(risiko.getAktiverSpieler()).size())) {
				try {
					landNr = Integer.parseInt(einlesen());
				} catch (NumberFormatException e) {
					landNr = 0;
					printError(e);
				}
			}
			landNr--;
			risiko.setzeEinheiten(risiko.getLaender(risiko.getAktiverSpieler()).elementAt(landNr),1);
		}
		risiko.weiter();
	}

	/**
	 * Lsst nach einem gewonnenen Land Einheiten nachrcken <i>(Spielphase: 70)</i>
	 * @throws RemoteException 
	 */
	public void nachruecken() throws RemoteException {
		print("Wie viele Einheiten sollen nachrcken? ");
		int anzEinheiten = -1;
		while(anzEinheiten<0) {
			try {
				anzEinheiten = Integer.parseInt(einlesen());
			} catch (NumberFormatException e) {
				anzEinheiten = -1;
				printError(e);
			}
		}
		try {
			risiko.verschiebeEinheiten(risiko.getAngreifer(), 
					risiko.getVerteidiger(), 
					anzEinheiten);
		} catch (NichtGenugEinheitenException e) {
			printError(e);
		} catch (LandBeteiligtException e) {
			printError(e);
		} catch (LandNichtImBesitzException e) {
			printError(e);
		} catch (NichtBenachbartException e) {
			printError(e);
		}
		risiko.weiter();
	}
	
	/**
	 * Fgt einen Log-Eintrag hinzu und sendet ihn an alle Clients
	 * 
	 * @param text einzufgender Text
	 * @throws RemoteException
	 */
	public void addLogEintrag(String text) throws RemoteException {
		String erg = "\n##########\n";
		erg += text+"\n";
		erg += "##########\n";
		risiko.notifyClients(erg);
	}
	
	/**
	 * Wartet :-)
	 */
	public void warten() {
		//	tue nichts
	}

	@Override
	public void update(Observable o, Object arg) {
		if(arg instanceof String) {
			String text = (String) arg;
			print(text);
		} 
	}



}
