package risiko.server.domain;

import java.io.FileNotFoundException;
import java.io.IOException;
import java.net.MalformedURLException;
import java.rmi.Naming;
import java.rmi.NotBoundException;
import java.rmi.RemoteException;
import java.util.Collections;
import java.util.Iterator;
import java.util.Observable;
import java.util.Random;
import java.util.Vector;

import risiko.common.exceptions.KeineNachbarnException;
import risiko.common.exceptions.LandBeteiligtException;
import risiko.common.exceptions.LandNichtImBesitzException;
import risiko.common.exceptions.NichtBenachbartException;
import risiko.common.exceptions.NichtGenugEinheitenException;
import risiko.common.exceptions.NichtGenugWuerfelException;
import risiko.common.exceptions.VerteidigerUebersteigtAngreiferException;
import risiko.common.interfaces.BefreiungsMission;
import risiko.common.interfaces.Kontinent;
import risiko.common.interfaces.LaenderMission;
import risiko.common.interfaces.Land;
import risiko.common.interfaces.Mission;
import risiko.common.interfaces.Notifiable;
import risiko.common.interfaces.SpielMgr;
import risiko.common.interfaces.Spieler;
import risiko.server.persistence.PersMgr;
import risiko.server.valueobjects.BefreiungsMissionImpl;
import risiko.server.valueobjects.LaenderMissionImpl;
import risiko.server.valueobjects.MissionImpl;

/**
 * Haptverwaltungsklasse des Spiels.
 * 
 * @author dgrosche
 * @version 2009-05-25
 *
 */
public class SpielMgrImpl extends Observable implements SpielMgr {
	
	private static final long serialVersionUID = 3334778711561069533L;
	
	private WeltMgr welt;
	private SpielerMgr spieler;
	private MissionsMgr missionen;
	
	private KampfMgr aktiverKampf;
	private KartenMgr karten;
	
	private PersMgr persistenz;
	
	private int phase, anzSpieler;
	private boolean spielstart;
	private Vector<Notifiable> clients;

	/**
	 * Konstruktor erstellt Unterverwaltungsklassen:
	 * <br>    - Weltverwaltung
	 * <br>    - Spielerverwaltung
	 * <br>    - Kartenverwaltung
	 * <br>    - Datenspeicherung
	 * 
	 */
	public SpielMgrImpl() throws RemoteException {
		persistenz = new PersMgr();
		try {
			welt = new WeltMgr(persistenz);
		} catch(Exception e) {
			System.err.println(e);
			welt = new WeltMgr();
		}
		spieler = new SpielerMgr();
		karten = new KartenMgr(welt.getLaenderliste().size());
		anzSpieler = 100;
		clients = new Vector<Notifiable>();
	}
	
	/*
	 * Spielstart 
	 */
	
	/* (non-Javadoc)
	 * @see risiko.server.domain.SpielMgrInt#addSpieler(java.lang.String)
	 */
	public int addSpieler(String name) throws RemoteException {
		spieler.addSpieler(name);
		int nr = spieler.getAllSpieler().size();
		System.out.println("Spieler: "+nr);
		return nr;
	}
	
	/* (non-Javadoc)
	 * @see risiko.server.domain.SpielMgrInt#verteileLaender()
	 */
	public void verteileLaender() throws RemoteException {
		//Alle Lnder werden in eine Liste nicht verteilter Lnder kopiert
		Vector<Land> freieLaender = new Vector<Land>(welt.getLaenderliste());
		while(freieLaender.size()!=0) {
			for(int i=0; i<spieler.getAllSpieler().size(); i++) {
				if(freieLaender.size()!=0) {
					//Es wird eine Zufallszahl zwischen 0 und der Anzahl nicht verteilter Lnder ermittelt
					Random random = new Random();
					int zahl = random.nextInt(freieLaender.size());
					//Dem Land wird der Besitzer zuegordnet
					freieLaender.elementAt(zahl).setBesitzer(spieler.getAllSpieler().elementAt(i));
					//Die Lnderanzahl des Spielers wird aktualisiert
					int x = spieler.getAllSpieler().elementAt(i).getAnzLaender();
					spieler.getAllSpieler().elementAt(i).setAnzLaender(x+1);
					//Das Land wird aus den freien Lndern genommen
					freieLaender.remove(zahl);
				}
			}
		}
	}
	
	/* (non-Javadoc)
	 * @see risiko.server.domain.SpielMgrInt#verteileMissionen()
	 */
	public void verteileMissionen() throws RemoteException {
		//Die Missionsverwaltung wird erstellt
		missionen = new MissionsMgr(welt, spieler.getAllSpieler());
		//Alle Missionen werden in eine Liste nicht verteilter Missionen kopiert
		Vector<MissionImpl> freieMissionen = new Vector<MissionImpl>(missionen.getMissionsliste());
		for(Spieler p : spieler.getAllSpieler()) {
			if(freieMissionen.size()!=0) {
				//Es wird eine Zufallszahl zwischen 0 und der Anzahl nicht verteilter Missionen ermittelt
				Random random = new Random();
				int zahl = random.nextInt(freieMissionen.size());
				Mission m = freieMissionen.elementAt(zahl);
				//Die Mission wird aus den freien Missionen genommen
				freieMissionen.removeElementAt(zahl);
				//Bei einer Befreiungsmission muss geguckt werden, ob sich der Spieler selbst verfolgen soll
				if(m instanceof BefreiungsMissionImpl) {
					if(((BefreiungsMission) m).getFeind().equals(p)) {
						//... wenn ja, dann muss eine Alternative gemacht werden
						LaenderMissionImpl alternative = new LaenderMissionImpl(24,1);
						alternative.setMessage("Befreien Sie 24 Lnder Ihrer Wahl!");
						alternative.setSpieler(p);
						p.setMission(alternative);
						//Die normale Mission wird gestrichen und die Alternative eingefgt
						missionen.getMissionsliste().remove(m);
						missionen.getMissionsliste().add(alternative);
					} else {
						//... wenn nicht, dann ist alles ok.
						m.setSpieler(p);
						p.setMission(m);
					}
				//Bei Lndermissionen, muss die Lnderliste des Spielers hinzugefgt werden
				} else if(m instanceof LaenderMissionImpl) {
					m.setSpieler(p);
					p.setMission(m);
					((LaenderMission) m).setLaenderliste(getLaender(p)); 
				} else {
					m.setSpieler(p);
					p.setMission(m);
				}
			}
		}
	}
	
	/* (non-Javadoc)
	 * @see risiko.server.domain.SpielMgrInt#weltErobern()
	 */
	public void weltErobern() throws RemoteException {
		missionen.getMissionsliste().removeAllElements();
		for(Spieler p : spieler.getAllSpieler()) {
			MissionImpl m = new LaenderMissionImpl(welt.getLaenderliste().size(),1);
			m.setMessage("Erobern Sie die Welt!");
			missionen.getMissionsliste().add(m);
			m.setSpieler(p);
			p.setMission(m);
		}
	}
	
	/* (non-Javadoc)
	 * @see risiko.server.domain.SpielMgrInt#reihenfolge()
	 */
	public Vector<Integer> reihenfolge() throws RemoteException {
		//Das Auswrfeln geschieht ber den KampfMgr
		Vector<Integer> zahlen = new Vector<Integer>();
		KampfMgr wuerfel = new KampfMgr();
		for(int i=0; i<spieler.getAllSpieler().size(); i++) {
			zahlen.add(wuerfel.wuerfel());
		}
		//Die hchste Zahl wird ermittelt
		int hoch = Collections.max(zahlen);
		//Der Spieler mit der hchsten Zahl ist dran
		int idx  = zahlen.indexOf(hoch);
		spieler.getAllSpieler().elementAt(idx).setDran(true);
		//Die Wrfelzahlen werden zur Ausgabe bergeben
		return zahlen;
	}
	
	/*
	 * Spielverlauf
	 */
	
	/* (non-Javadoc)
	 * @see risiko.server.domain.SpielMgrInt#weiter()
	 */
	public void weiter() throws RemoteException {
		if(isGewonnen()) this.setPhase(100);
		switch(phase) {
			// 1: Neuer Angriff 
			case 1: this.setPhase(50); break; //weiter bei 50
			// 2: Einheiten verschieben
			case 2: this.setPhase(3); break; //weiter bei 3
			// 3: Zug beenden
			case 3: {
				this.beendeZug();
				//fallse mehr als 4 Karten
				if(getAktiverSpieler().getKarten().size()>4)
					this.setPhase(5);	//weiter bei 5
				else
					this.setPhase(60);	//weiter bei 60
			} break;
			// 4: Lnder anzeigen
			case 4: this.setPhase(0); break; //weiter bei 0 (Hauptmen)
			// 5: Karten einlsen
			case 5: this.setPhase(0); break; //weiter bei 0 (Hauptmen)
			// 6: Spiel speichern
			case 6: this.setPhase(0); break; //weiter bei 0 (Hauptmen)
			// 7: Spiel laden
			case 7: this.setPhase(0); break; //weiter bei 0 (Hauptmen)
			// 50: Verteidigen
			case 50: {
				if(aktiverKampf.getAngreifer().getBesitzer().equals(
				   aktiverKampf.getVerteidiger().getBesitzer()))
						this.setPhase(70);  // weiter bei 70
				else
						this.setPhase(0);	//weiter bei 0 (Hauptmen)
			} break;
			// 60: Verstrkung aufstellen
			case 60: this.setPhase(0); break; //weiter bei 0 (Hauptmen)
			// 70: Zusatzeinheiten nachrcken
			case 70: this.setPhase(0); break;
		}
	}
	
	/* (non-Javadoc)
	 * @see risiko.server.domain.SpielMgrInt#setPhase(int)
	 */
	public void setPhase(int wert) {
		this.phase = wert;
	}
	
	/* (non-Javadoc)
	 * @see risiko.server.domain.SpielMgrInt#getPhase()
	 */
	public int getPhase() {
		return this.phase;
	}
	
	/* (non-Javadoc)
	 * @see risiko.server.domain.SpielMgrInt#starteAngriff(risiko.common.valueobjects.Land, risiko.common.valueobjects.Land, int)
	 */
	public void starteAngriff(Land a, Land b, int x) throws LandNichtImBesitzException, NichtBenachbartException, NichtGenugEinheitenException, NichtGenugWuerfelException, RemoteException {
		//Ein neuer Kampf wird erstellt
		if(a.getBesitzer().getNummer()!=getAktiverSpieler().getNummer()) {
			throw new LandNichtImBesitzException(a, this.getAktiverSpieler());
		}
		if(!(welt.isNachbar(a, b))) {
			throw new NichtBenachbartException(a,b);
		}
		if(a.getAnzEinheiten()==1) {
			throw new NichtGenugEinheitenException(a);
		}
		this.aktiverKampf = new KampfMgr();
		if(x>3) {
			throw new NichtGenugWuerfelException(x,3,this.getWuerfelAngreifer());
		}
		//Angreifer, Verteidiger und Angriffseinheiten werden gesetzt
		aktiverKampf.setAngreifer(a);
		aktiverKampf.setVerteidiger(b);
		aktiverKampf.setEinheitenAngriff(x);
	}
	
	/* (non-Javadoc)
	 * @see risiko.server.domain.SpielMgrInt#verschiebeEinheiten(risiko.common.valueobjects.Land, risiko.common.valueobjects.Land, int)
	 */
	public void verschiebeEinheiten(Land a, Land b, int x) throws NichtGenugEinheitenException, LandBeteiligtException, LandNichtImBesitzException, NichtBenachbartException, RemoteException {
		//Wenn Spiel in Verschiebe-Phase
		if(this.getPhase()==2) {
			if(a.getBesitzer().getNummer()!=getAktiverSpieler().getNummer()) {
				throw new LandNichtImBesitzException(a, this.getAktiverSpieler());
			}
			if(b.getBesitzer().getNummer()!=getAktiverSpieler().getNummer()) {
				throw new LandNichtImBesitzException(b, this.getAktiverSpieler());
			}
			if(!(welt.isNachbar(a, b))) {
				throw new NichtBenachbartException(a,b);
			}
			//Prfe erst ob eines der Lnder schon beteiligt war
			if(a.isBeteiligt()) throw new LandBeteiligtException(a);
			if(b.isBeteiligt()) throw new LandBeteiligtException(b);
		}
		//Dann verschiebe
		welt.verschiebeEinheiten(a, b, x);
	}

	/* (non-Javadoc)
	 * @see risiko.server.domain.SpielMgrInt#beendeZug()
	 */
	public void beendeZug() throws RemoteException {
		karten.addKarte(this.getAktiverSpieler());
		spieler.beendeZug();
		karten.setLandErobert(false);
		welt.setUnbeteiligt();
	}
	
	
	/* (non-Javadoc)
	 * @see risiko.server.domain.SpielMgrInt#kartenEinloesen(int[])
	 */
	public void kartenEinloesen(int[] kartensatz) throws RemoteException {
		karten.kartenEinloesen(kartensatz, this.getAktiverSpieler());
	}
	
	/* (non-Javadoc)
	 * @see risiko.server.domain.SpielMgrInt#speichern()
	 */
	public String speichern() {
		String erg;
		//Fge zu speichernde Objekte hinzu
		Vector<Object> objekte = new Vector<Object>();
		objekte.add(welt);
		objekte.add(spieler);
		try {
			//Speichere ber Dateiverwaltung
			persistenz.speichern(objekte);
			erg = "Speichern erfolgreich.";
		} catch (FileNotFoundException e) {
			erg = "FEHLER: "+e;
		} catch (IOException e) {
			erg = "FEHLER: "+e;
		}
		return erg;
	}
	
	/* (non-Javadoc)
	 * @see risiko.server.domain.SpielMgrInt#laden()
	 */
	public String laden()  {
		String erg;
		try {
			//Lade Welt- und Spielerverwaltung aus der Persistenz
			persistenz.laden();
			welt = persistenz.getWelt();
			spieler = persistenz.getSpieler();
			//Wenn keine Missionsverwaltung vorhanden...
			if(missionen==null) {
				//... lege eine neue an
				missionen = new MissionsMgr(welt, spieler.getAllSpieler());
				//... setze die Spieler entsprechend
				for(Spieler p : spieler.getAllSpieler()) {
					for(Mission m : missionen.getMissionsliste()) {
						if(m.equals(p.getMission())) {
							m.setSpieler(p);
							p.setMission(m);
						}
					}
				}
			}
			erg = "Laden erfolgreich.";
		} catch (FileNotFoundException e) {
			erg = "FEHLER: "+e;
		} catch (IOException e) {
			erg = "FEHLER: "+e;
		} catch (ClassNotFoundException e) {
			erg = "FEHLER: "+e;
		}

		return erg;
	}
	
	/* (non-Javadoc)
	 * @see risiko.server.domain.SpielMgrInt#verteidige(int)
	 */
	public void verteidige(int y) throws NichtGenugWuerfelException, VerteidigerUebersteigtAngreiferException, NichtGenugEinheitenException, RemoteException {
		aktiverKampf.setEinheitenVerteidigung(y);
		if(aktiverKampf.angriff()) {
			this.karten.setLandErobert(true);
			aktiverKampf.getAngreifer().setBesitzer(spieler.getAktiverSpieler());
			aktiverKampf.getVerteidiger().setBesitzer(spieler.getAktiverSpieler());
		}
	}
	
	/* (non-Javadoc)
	 * @see risiko.server.domain.SpielMgrInt#getVerstaerkung()
	 */
	public int getVerstaerkung() throws RemoteException {
		Spieler p = this.getAktiverSpieler();
		//Anzahl der Lnder geteilt durch 3
		int erg = p.getAnzLaender() / 3;	
		//Zusatzpunkte fr eroberten Kontinenten
		for(int i=0; i<welt.getKontinentenliste().size(); i++) {
			Kontinent k = welt.getKontinentenliste().elementAt(i);
			if(welt.kontinentErobert(k)) {
				if(k.getLaenderListe().elementAt(0).getBesitzer().equals(getAktiverSpieler())) {
					erg += k.getPunkt();
				}
			}
		}
		//Zusatzpunkte durch in Vorrunde eingelste Karten
		erg += p.getKartenBonus();
		p.setKartenBonus(0);
		//Wenn alles zusammen, kleiner als 3 wre, setze es auf 3.
		erg = (erg<3) ? 3 : erg;
		return erg;
	}		

	/*
	 * Getter, Setter und weitere Methoden
	 */
	
	/* (non-Javadoc)
	 * @see risiko.server.domain.SpielMgrInt#pruefKarten()
	 */
	public boolean pruefKarten() throws RemoteException {
		return karten.pruefKarten(this.getAktiverSpieler());
	}
		
	/* (non-Javadoc)
	 * @see risiko.server.domain.SpielMgrInt#getEinloesbareKarten()
	 */
	public Vector<int[]> getEinloesbareKarten() {
		return karten.getEinloesbareKarten();
	}
	
	/* (non-Javadoc)
	 * @see risiko.server.domain.SpielMgrInt#isGewonnen()
	 */
	public boolean isGewonnen() throws RemoteException {
		Spieler p = this.getAktiverSpieler();
		//Fr Lndermissionen wird die Lnderliste aktualisiert
		missionen.setLaenderliste(p.getMission(), getLaender(p));
		missionen.pruefMissionen();
		if(p.getMission().isErfuellt()) {
			return true;
		} else {
			return false;
		}
	}
		
	/* (non-Javadoc)
	 * @see risiko.server.domain.SpielMgrInt#getVerteidiger()
	 */
	public Land getVerteidiger() {
		if(aktiverKampf!=null) {
			return aktiverKampf.getVerteidiger();
		} else {
			return null;
		}
	}
	
	/* (non-Javadoc)
	 * @see risiko.server.domain.SpielMgrInt#getAngreifer()
	 */
	public Land getAngreifer() {
		if(aktiverKampf!=null) {
			return aktiverKampf.getAngreifer();
		} else {
			return null;
		}
	}
	
	/* (non-Javadoc)
	 * @see risiko.server.domain.SpielMgrInt#getWuerfelAnfreifer()
	 */
	public Vector<Integer> getWuerfelAngreifer() {
		if(aktiverKampf!=null) {
			return aktiverKampf.getWuerfelAngreifer();
		} else {
			return null;
		}
	}
	
	/* (non-Javadoc)
	 * @see risiko.server.domain.SpielMgrInt#getWuerfelVerteidiger()
	 */
	public Vector<Integer> getWuerfelVerteidiger() {
		if(aktiverKampf!=null) {
			return aktiverKampf.getWuerfelVerteidiger();
		} else {
			return null;
		}
	}
	
	/* (non-Javadoc)
	 * @see risiko.server.domain.SpielMgrInt#getEinheitenAngriff()
	 */
	public int getEinheitenAngriff() {
		if(aktiverKampf!=null) {
			return aktiverKampf.getEinheitenAngriff();
		} else {
			return -1;
		}
	}
	
	/* (non-Javadoc)
	 * @see risiko.server.domain.SpielMgrInt#getAktiverSpieler()
	 */
	public Spieler getAktiverSpieler() throws RemoteException {
		return spieler.getAktiverSpieler();
	}
	
	/* (non-Javadoc)
	 * @see risiko.server.domain.SpielMgrInt#getAllSpieler()
	 */
	public Vector<Spieler> getAllSpieler() {
		return spieler.getAllSpieler();
	}

	
	/* (non-Javadoc)
	 * @see risiko.server.domain.SpielMgrInt#getAllLaender()
	 */
	public Vector<Land> getAllLaender() {
		return this.welt.getLaenderliste();
	}
	
	/* (non-Javadoc)
	 * @see risiko.server.domain.SpielMgrInt#getLaender(risiko.common.valueobjects.Spieler)
	 */
	public Vector<Land> getLaender(Spieler spieler) throws RemoteException {
		Vector<Land> erg = new Vector<Land>();
		for(int i=0; i<welt.getLaenderliste().size(); i++) {
			if(welt.getLaenderliste().elementAt(i).getBesitzer().getNummer()==spieler.getNummer()) {
				erg.add(welt.getLaenderliste().elementAt(i));
			}
		}
		return erg;
	}

	/* (non-Javadoc)
	 * @see risiko.server.domain.SpielMgrInt#getNachbarn(risiko.common.valueobjects.Land)
	 */
	public Vector<Land> getNachbarn(Land a) throws RemoteException {
		return this.welt.getNachbarn(a);
	}
	
	/* (non-Javadoc)
	 * @see risiko.server.domain.SpielMgrInt#getEigeneNachbarn(risiko.common.valueobjects.Land)
	 */
	public Vector<Land> getEigeneNachbarn(Land a) throws KeineNachbarnException, RemoteException {
		Vector<Land> erg = new Vector<Land>();
		Iterator<Land> iter = this.getNachbarn(a).iterator();
		while(iter.hasNext()) {
			Land einLand = iter.next();
			if(einLand.getBesitzer().equals(a.getBesitzer())) {
				erg.add(einLand);
			}
		}
		if(erg.size()==0) throw new KeineNachbarnException(a,false);
		return erg;
	}
	
	/* (non-Javadoc)
	 * @see risiko.server.domain.SpielMgrInt#getFremdeNachbarn(risiko.common.valueobjects.Land)
	 */
	public Vector<Land> getFremdeNachbarn(Land a) throws KeineNachbarnException, RemoteException {
		Vector<Land> erg = new Vector<Land>();
		Iterator<Land> iter = this.getNachbarn(a).iterator();
		while(iter.hasNext()) {
			Land einLand = iter.next();
			if(!(einLand.getBesitzer().equals(a.getBesitzer()))) {
				erg.add(einLand);
			}
		}
		if(erg.size()==0) throw new KeineNachbarnException(a,true);
		return erg;
	}
	
	/* (non-Javadoc)
	 * @see risiko.server.domain.SpielMgrInt#isNachbar(risiko.common.valueobjects.Land, risiko.common.valueobjects.Land)
	 */
	public boolean isNachbar(Land a, Land b) throws RemoteException {
		if(!(a.equals(b))) {
			return welt.isNachbar(a, b);
		} else {
			return false;
		}
	}
	
	/* (non-Javadoc)
	 * @see risiko.server.domain.SpielMgrInt#setzeEinheiten(risiko.common.valueobjects.Land, int)
	 */
	public void setzeEinheiten(Land l, int x) throws RemoteException {
		welt.addEinheiten(l, x);	
	}

	/* (non-Javadoc)
	 * @see risiko.server.domain.SpielMgrInt#setAnzSpieler(int anzSpieler)
	 */
	public void setAnzSpieler(int anzSpieler) {
		this.anzSpieler = anzSpieler;
	}

	/* (non-Javadoc)
	 * @see risiko.server.domain.SpielMgrInt#getAnzSpieler()
	 */
	public int getAnzSpieler() {
		return anzSpieler;
	}

	/* (non-Javadoc)
	 * @see risiko.server.domain.SpielMgrInt#setSpielstart()
	 */
	public void setSpielstart(boolean spielstart) {
		this.spielstart = spielstart;
	}

	/* (non-Javadoc)
	 * @see risiko.server.domain.SpielMgrInt#getSpielstart()
	 */
	public boolean isSpielstart() {
		return spielstart;
	}
	
	/* (non-Javadoc)
	 * @see risiko.server.domain.SpielMgrInt#addClient()
	 */
	public void addClient(String servicename) throws MalformedURLException, RemoteException, NotBoundException {
		Notifiable n = (Notifiable) Naming.lookup(servicename);
		clients.add(n);
	}

	/* (non-Javadoc)
	 * @see risiko.server.domain.SpielMgrInt#notifyClients(Object arg)
	 */
	public void notifyClients(Object arg) throws RemoteException {
		for(Notifiable n : clients) {
			n.update(arg);
		}
	}

}
