/**
 * 
 */
package risiko.server.domain;

import java.io.Serializable;
import java.rmi.RemoteException;
import java.util.Collections;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Random;
import java.util.Set;
import java.util.Vector;

import risiko.common.interfaces.Spieler;

/**
 * Verwaltet die Kartenfunktionen
 * 
 * @author dgrosche
 * @version 2009-05-25
 *
 */
public class KartenMgr implements Serializable {
	
	private static final long serialVersionUID = -7419580734530898663L;
	private Vector<Integer> kartenstapel;
	private Vector<Integer> eingeloesteKarten;
	private Vector<int[]> einloesbareKarten;
	private boolean landErobert;
	private int einloeseNummer;
	
	/**
	 * Konstruktor:
	 * <br>   - erstellt Kartenstapel, Ablagestapel und Stapel fr einlsbare Karten
	 * <br>   - setzt die Anzahl bisher eingelsten Karten auf 0
	 * <br>   - fllt Kartenstapel mit je einer Karte fr ein Land gliechmig mit
	 *          Soldaten, Reitern und Kanonen
	 * <br>   - fgt zwei Joker zum Kartenstapel hinzu
	 * <br>   - mischt den Kartenstapel
	 *    
	 * @param laenderanzahl Anzahl aller Lnder = Anzahl der Karten
	 */
	public KartenMgr(int laenderanzahl) {
		kartenstapel = new Vector<Integer>();
		eingeloesteKarten = new Vector<Integer>();
		setEinloesbareKarten(new Vector<int[]>());
		setLandErobert(false);
		einloeseNummer = 0;
		int max = laenderanzahl / 3;
		for(int i=0; i<max; i++) {
			kartenstapel.add(0);
			kartenstapel.add(1);
			kartenstapel.add(2);
		}
		int rest = laenderanzahl % 3;
		for(int i=0; i<rest; i++) {
			kartenstapel.add(new Random().nextInt(3));
		}
		kartenstapel.add(3);
		kartenstapel.add(3);
		Collections.shuffle(kartenstapel);
	}
	
	/**
	 * <br>Prft ob Karten eingelst werden knnen
	 * <br>Fgt alle einlsbaren Kartenstze zu einlsbaren Karten hinzu
	 * 
	 * @return true, wenn Karten eingelst werden knnen.
	 * @throws RemoteException 
	 */
	public boolean pruefKarten(Spieler p) throws RemoteException {
		Vector<Integer> karten = new Vector<Integer>(p.getKarten());
		Vector<int[]> erg = new Vector<int[]>();
		boolean rueck = false;
		if(karten.size()<3) return false;
		
		// 1. Mglichkeit: 3 unterschiedliche Karten
		Set<Integer> menge = new HashSet<Integer>(karten);
		if(menge.size()==3) { // 3 Unterschiedliche
			rueck = true;
			Iterator<Integer> iter = menge.iterator();
			int[] kartensatz = new int[3];
			int i = 0;
			while(iter.hasNext()) {
				kartensatz[i] = iter.next();
				i++;	
			}
			erg.add(kartensatz);
		}
		if(menge.size()==4) { // 4 Unterschiedliche
			int[] k1 = {0, 1, 2};
			erg.add(k1);
			int[] k2 = {0, 1, 3};
			erg.add(k2);
			int[] k3 = {0, 2, 3};
			erg.add(k3);
			int[] k4 = {1, 2, 3};
			erg.add(k4);
			rueck = true;
		}
		
		// 2. Mglichkeit: 3 gleiche Karten
		Collections.sort(karten);
		int anzahl = 0;
		while(karten.size()>0) {
			anzahl = 0;
			int wert = karten.firstElement();
			for(int i=0; i<karten.size(); i++) {
				if(wert==karten.elementAt(i)) {
					anzahl++;
				}
			}
			while(karten.lastIndexOf(wert)!=-1) {
				karten.removeElement(wert);
			}
			if(anzahl>2) {
				int[] kartensatz = {wert, wert, wert};
				erg.add(kartensatz);
				rueck = true;
			}
		}
		
		// 3. Mglichkeit: 2 gleiche Karten und 1 Joker
		karten = new Vector<Integer>(p.getKarten());
		if(karten.contains(3)) {
			Collections.sort(karten);
			anzahl = 0;
			while(karten.size()>0) {
				anzahl = 0;
				int wert = karten.firstElement();
				for(int i=0; i<karten.size(); i++) {
					if(wert==karten.elementAt(i)) {
						anzahl++;
					}
				}
				while(karten.lastIndexOf(wert)!=-1) {
					karten.removeElement(wert);
				}
				if(anzahl>1) {
					int[] kartensatz = {wert, wert, 3};
					erg.add(kartensatz);
					rueck = true;
				}
			}
		}
		
		
		this.setEinloesbareKarten(erg);
		return rueck;
	}
	
	/**
	 * Lscht einzuloesende Karten des Spielers, setzt Einlsenummer hoch und schreibt
	 * dem Spieler die Bonuspunkte gut
	 * 
	 * @param kartensatz einzulsender Kartensatz
	 * @param p Spieler, der Karten einlst
	 * @throws RemoteException 
	 */
	public void kartenEinloesen(int[] kartensatz, Spieler p) throws RemoteException {
		this.einloeseNummer++;
		for(int i=0; i<kartensatz.length; i++) {
			eingeloesteKarten.add(kartensatz[i]);
			p.getKarten().removeElement(kartensatz[i]);
		}
		switch(einloeseNummer) {
			case 1: p.setKartenBonus(4); break; //4 Extra-Armeen
			case 2: p.setKartenBonus(6); break; //6 Extra-Armeen
			case 3: p.setKartenBonus(8); break; //8 Extra-Armeen
			case 4: p.setKartenBonus(10); break; //10 Extra-Armeen
			case 5: p.setKartenBonus(12); break; //12 Extra-Armeen
			case 6: p.setKartenBonus(15); break; //15 Extra-Armeen
			default: {
				int extra = einloeseNummer - 6;
				extra *= 5;
				extra += 15;
				p.setKartenBonus(extra);
			} break;
		}
	}
	
	/**
	 * Nimmt eine Karte vom Kartenstapel und fgt sie dem Spieler hinzu
	 * 
	 * @param p Spieler der die Karte bekommt
	 * @throws RemoteException 
	 */
	public void addKarte(Spieler p) throws RemoteException {
		if(this.isLandErobert()) {
			if(kartenstapel.isEmpty()) {
				Collections.shuffle(eingeloesteKarten);
				kartenstapel = new Vector<Integer>(eingeloesteKarten);
				eingeloesteKarten.removeAllElements();
			}
			int karte = kartenstapel.firstElement();
			kartenstapel.removeElementAt(0);
			p.addKarte(karte);
		}
	}

	/**
	 * @param einloesbareKarten the einloesbareKarten to set
	 */
	public void setEinloesbareKarten(Vector<int[]> einloesbareKarten) {
		this.einloesbareKarten = einloesbareKarten;
	}

	/**
	 * @return the einloesbareKarten
	 */
	public Vector<int[]> getEinloesbareKarten() {
		return einloesbareKarten;
	}

	/**
	 * @param landErobert the landErobert to set
	 */
	public void setLandErobert(boolean landErobert) {
		this.landErobert = landErobert;
	}

	/**
	 * @return the landErobert
	 */
	public boolean isLandErobert() {
		return landErobert;
	}

}
