package risiko.local.ui.gui;

import java.awt.CardLayout;
import java.awt.Color;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.MouseEvent;
import java.util.Vector;

import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JSpinner;
import javax.swing.JTextArea;
import javax.swing.JTextField;
import javax.swing.SpinnerNumberModel;
import javax.swing.event.MouseInputAdapter;

import risiko.local.domain.SpielMgr;
import risiko.local.domain.exceptions.KeineNachbarnException;
import risiko.local.domain.exceptions.LandBeteiligtException;
import risiko.local.domain.exceptions.LandNichtImBesitzException;
import risiko.local.domain.exceptions.NichtBenachbartException;
import risiko.local.domain.exceptions.NichtGenugEinheitenException;
import risiko.local.domain.exceptions.NichtGenugWuerfelException;
import risiko.local.domain.exceptions.VerteidigerUebersteigtAngreiferException;
import risiko.local.valueobjects.Land;

/**
 * Der Event-Manager reagiert auf smtliche Events und implementiert 
 * damit alle notwendigen Listener. Die Klasse legt die Schnittstelle
 * zum Spiel-Manager fest und dient daher als Hauptklasse der GUI
 * (allerdings nicht als Hauptprogrammklasse)
 * 
 * @author dgrosche
 * @version 2009-06-25
 *
 */
public class EventMgr extends MouseInputAdapter implements ActionListener {

	private Fenster gui;
	private SpielMgr risiko;
	private int klickStatus;
	private Land ursprung, ziel;
	
	/**
	 * Konstruktor erstelle ein neuen Event-Manager
	 * @param frame Fenster, dass die Listener implementiert
	 */
	protected EventMgr(Fenster frame) {
		gui = frame;
		risiko = new SpielMgr();
		klickStatus = 0;
		
		/*
		 * Popupfenster zu Beginn legt fest ob ein neues Spiel gestartet oder ein altes
		 * geladen wird
		 */
		Object[] options = {"Neues Spiel", "Spiel laden"};
		int n = JOptionPane.showOptionDialog(Fenster.frame,
				"Neues Spiel starten?",
				"Spielstart",
				JOptionPane.YES_NO_OPTION,
				JOptionPane.QUESTION_MESSAGE,
				null,     //do not use a custom Icon
				options,  //the titles of buttons
				options[0]); //default button title
		
		if(n==0) {
			this.neuesSpiel();
			this.spielerMenu();
		}
		if(n==1) this.addLogEintrag(risiko.laden());
		
		
		Vector<Land> laender = new Vector<Land>(risiko.getAllLaender());
		WeltPanel.setLaender(laender);
		WeltPanel.zeichneLaender();
		
		if(n==1) {
			this.beendeZug(true);
			risiko.setPhase(0);
			this.spielerMenu();
		}
	}
	
	/**
	 * Initialisiert das Spiel, d.h.
	 * <ul>
	 * <li>		Spieler werden registriert			</li>
	 * <li>		Lnder werden verteilt				</li>
	 * <li>		Startreihenfolge wird festgelegt	</li>
	 * <li>		Anfangseinheiten werden gesetzt		</li>
	 * </ul>
	 */
	private void neuesSpiel() {
		//Anzahl der Spieler wird bestimmt. Die Zahl muss zwischen 1 und 6 liegen
		String input="";
		while(input.equals("")) {
			Object[] possibilities = {"3","4","5","6"};
			input = (String) JOptionPane.showInputDialog(
					Fenster.frame,
					"Anzahl der Spieler:",
					"Anzahl der Spieler:",
					JOptionPane.QUESTION_MESSAGE,
					null,
					possibilities,
					"3");
			input = (input==null) ? "" : input; 
		}

		int anzSpieler = Integer.parseInt(input);
		
		//Spieler werden hinzugefgt
		for(int i=1; i<=anzSpieler; i++) {
			String name = "";
			while (name.equals("")) {
				name = (String) JOptionPane.showInputDialog(
						Fenster.frame,
						"Name:",
						"Spieler "+i+":",
						JOptionPane.QUESTION_MESSAGE,
						null,
						null,
						"");
				name = (name==null) ? "" : name; 
			}
			risiko.addSpieler(name);
		}
		
		//Spieler erhalten Lnder und Missioen
		risiko.verteileLaender();
		risiko.verteileMissionen();
		
		//Missionen oder Welt erobern
		Object[] options = {"Missionen ausfhren", "Welt erobern"};
		int auswahl = JOptionPane.showOptionDialog(Fenster.frame,
				"Spielmodus whlen:",
				"Spielmodus",
				JOptionPane.YES_NO_OPTION,
				JOptionPane.QUESTION_MESSAGE,
				null,     //do not use a custom Icon
				options,  //the titles of buttons
				options[0]); //default button title
		if(auswahl==1) risiko.weltErobern();
		
		//Die Startreihenfolge wird ausgewrfelt
		Vector<Integer> zahlen = risiko.reihenfolge();
		String startreihenfolge = "Fr die Startreihenfolge wurden folgende Zahlen gewrfelt:\n";
		for(int i=0; i<zahlen.size(); i++) {
			startreihenfolge += "   "+risiko.getAllSpieler().elementAt(i)+": "+zahlen.elementAt(i)+"\n";
			risiko.beendeZug();
		}
		this.addLogEintrag(startreihenfolge);
		
		//Der erste Spieler beginnt mit dem Setzen von zustzlichen Einheiten
		this.beendeZug(true);
	}
	
	/**
	 * Gibt das Spielphasen entsprechende Men aus:
	 * <br>
	 *   <br><b> Phase 0:   </b> Hauptmen					
	 *   <br><b> Phase 1:   </b> Angriffsmen				
	 *   <br><b> Phase 2:   </b> Verschiebenmen			
	 *   <br><b> Phase 3:   </b> Lnder drucken			
	 *   <br><b> Phase 4:   </b> Zug beenden				
	 *   <br><b> Phase 5:   </b> Karten einlsen			
	 *   <br><b> Phase 6:   </b> Spiel speichern			
	 *   <br><b> Phase 7:   </b> Spiel laden				
	 *   <br><b> Phase 50:  </b> Verteidigen				
	 *   <br><b> Phase 60:  </b> Verstrkung setzen		
	 *   <br><b> Phase 70:  </b> Einheiten nachrcken		
	 *   <br><b> Phase 100: </b> Spiel beenden			
	 *   
	 */
	private void spielerMenu() {
		switch (risiko.getPhase()) {
			case 0: hauptmenu(); break; 		
			case 1: neuerAngriff(); break;		
			case 2: verschieben(); break;		
			case 3: beendeZug(false); break;		
//			case 4: printLaender(); break; 		
			case 5: kartenEinloesen(true); break;
			case 6: this.addLogEintrag(risiko.speichern()); this.beendeZug(false); break;
			case 7: laden(); break;
			case 50: verteidige(); break;		
			case 60: verstaerkung(); break;		
			case 70: nachruecken(); break;
			case 100: {
				warten();
				JOptionPane.showMessageDialog(Fenster.frame,
					    "Du hast gewonnen!",
					    "Gewonnen",
					    JOptionPane.INFORMATION_MESSAGE);
				WeltPanel.aktualisiere(); break;
			}
			default: warten(); break;
		}
	}
	
	/**
	 * Gibt das Hauptmen eines Spielers aus <i>(Spielphase: 0)</i>
	 */
	private void hauptmenu() {
		//Hauptmen anzeigen
		this.reset();
		WeltPanel.aktualisiere();
		JPanel menuPanel = (JPanel) gui.wichtigeKomponenten.get("MenuPanel");
		CardLayout cards = (CardLayout) menuPanel.getLayout();
		cards.show(menuPanel, "Haupt");
	}
	
	/**
	 * Zeigt Angriffspanel an <i>(Spielphase: 1)</i>
	 */
	private void neuerAngriff() {
		WuerfelPanel.reset();
		this.reset();
		JPanel menuPanel = (JPanel) gui.wichtigeKomponenten.get("MenuPanel");
		CardLayout cards = (CardLayout) menuPanel.getLayout();
		cards.show(menuPanel, "Angriff");
		MenuPanel.starten.get("Angriff").setEnabled(false);
		
		this.addLogEintrag("Whle dein Angriffsland:");
		klickStatus = 1;
	}
	
	/**
	 * Startet neuen Kampf <i>(Spielphase: 1)</i>
	 */
	private void starteAngriff() {
		JSpinner anzahl = (JSpinner) MenuPanel.anzahl.get("Angriff");
		int anz = (Integer) anzahl.getValue();
		try {
			risiko.getFremdeNachbarn(ursprung);
			risiko.starteAngriff(ursprung, ziel, anz);
			this.addLogEintrag("Angriff von "+ursprung+" nach "+ziel+" mit "+anz+" Angreifern.");
			risiko.weiter();
		} catch (LandNichtImBesitzException e) {
			this.setFehlerEintrag(e);
			risiko.setPhase(0); //Es kann nicht angegriffen werden, also Hauptmen
		} catch (KeineNachbarnException e) {
			this.setFehlerEintrag(e);
			risiko.setPhase(0); //Es kann nicht angegriffen werden, also Hauptmen 
		} catch (NichtBenachbartException e) {
			this.setFehlerEintrag(e);
			risiko.setPhase(0); //Es kann nicht angegriffen werden, also Hauptmen
		} catch (NichtGenugEinheitenException e) {
			this.setFehlerEintrag(e);
			risiko.setPhase(0); //Es kann nicht angegriffen werden, also Hauptmen
		} catch (NichtGenugWuerfelException e) {
			this.setFehlerEintrag(e);
		} finally {
			this.spielerMenu();
		}
	}
	
	/**
	 * Zeigt Transferpanel an <i>(Spielphase: 2)</i>
	 */
	private void verschieben() {
		this.reset();
		JPanel menuPanel = (JPanel) gui.wichtigeKomponenten.get("MenuPanel");
		CardLayout cards = (CardLayout) menuPanel.getLayout();
		cards.show(menuPanel, "Transfer");
		MenuPanel.starten.get("Transfer").setEnabled(false);
		
		this.addLogEintrag("Whle dein Ursprungsland:");
		klickStatus = 1;
	}
	
	/**
	 * Verschiebt Einheiten <i>(Spielphase: 2)</i>
	 */
	private void starteTransfer() {
		JSpinner anzahl = (JSpinner) MenuPanel.anzahl.get("Transfer");
		int anz = (Integer) anzahl.getValue();
		try {
			risiko.getEigeneNachbarn(ursprung);
			risiko.verschiebeEinheiten(ursprung, ziel, anz);
			this.addLogEintrag("Verschieben von "+anz+" Einheiten aus "+ursprung+" nach "+ziel);
		} catch (LandNichtImBesitzException e) {
			this.setFehlerEintrag(e);
		} catch (KeineNachbarnException e) {
			this.setFehlerEintrag(e);	
		} catch (NichtGenugEinheitenException e) {
			this.setFehlerEintrag(e);	
		} catch (LandBeteiligtException e) {
			this.setFehlerEintrag(e);
		} catch (NichtBenachbartException e) {
			this.setFehlerEintrag(e);
		}
		int nochmal=-1;
		if(risiko.getPhase()==2) {
			nochmal = JOptionPane.showOptionDialog(Fenster.frame,
					"Nochmal verschieben?",
					"Verschieben:",
					JOptionPane.YES_NO_OPTION,
					JOptionPane.QUESTION_MESSAGE,
					null,     //do not use a custom Icon
					null,  //the titles of buttons
					null); //default button title
		} else {
			nochmal = 1;
		}
		if(nochmal==1) {
			WeltPanel.aktualisiere();
			risiko.weiter();
		}
		this.spielerMenu();
	}
	
	/**
	 * Beendet den Zug, setzt den nchsten Spieler dran und aktualisiert die Panels <i>Spielphase: 3</i><br>
	 * @param start Wird die Methode beim Spielstart oder im Laufe des Spiels aufgerufen
	 */
	private void beendeZug(boolean start) {
		if(start) risiko.setPhase(60);
		else	  risiko.weiter(); 
		
		WuerfelPanel.reset();
		this.reset();
		String farbe = "";
		switch(risiko.getAktiverSpieler().getNummer()) {
			case 0: farbe = " (blau) "; break;
			case 1: farbe = " (gelb) "; break;
			case 2: farbe = " (grn) "; break;
			case 3: farbe = " (lila) "; break;
			case 4: farbe = " (orange) "; break;
			case 5: farbe = " (rot) "; break;
		}
		this.addLogEintrag(risiko.getAktiverSpieler()+farbe+" ist dran.");
		
		//Mission ausgeben
		JTextArea mission = (JTextArea) gui.wichtigeKomponenten.get("Mission");
		mission.setText("");
		this.logischeTextEingabe(risiko.getAktiverSpieler().getMission().toString(),mission);
		
		//Karten anzeigen
		JTextArea karten = (JTextArea) gui.wichtigeKomponenten.get("Karten");
		String text = "";
		karten.setText("");
		for(int i : risiko.getAktiverSpieler().getKarten()) {
			//Die Karten sind eigentlich nur Zahlen, entsprechend dieser Liste
			switch(i) {
				case 0: text+= ("   Reiter\n"); break;
				case 1: text+= ("   Kanonen\n"); break;
				case 2: text+= ("   Soldaten\n"); break;
				case 3: text+= ("   Joker\n"); break;
			}
		}
		this.logischeTextEingabe(text, karten);
		
		this.spielerMenu();
	}
	
	/**
	 * <br>Gibt Karten zurck. 
	 * <br>Prft ob einlsbare Karten vorhanden sind.
	 * <br>Lsst Karten einlsen.
	 * <br><i>(Spielphase: 5)</i> 
	 * @param pflicht Wenn mehr als 4 Karten vorhanden
	 */
	private void kartenEinloesen(boolean pflicht) {
		if(!risiko.pruefKarten()) {
			JOptionPane.showMessageDialog(Fenster.frame,
				    "Du kannst keine Karten eintauschen!",
				    "Karten eintauschen",
				    JOptionPane.WARNING_MESSAGE);
		} else {
			Vector<Object> options = new Vector<Object>();
			Vector<int[]> karten = risiko.getEinloesbareKarten(); 
			if(!pflicht) {
				options.add("0) keine Karten eintauschen!");
			}
			for(int i=0; i<karten.size(); i++) {
				String text = (i+1)+") ";
				for(int j=0; j<3; j++) {
					switch(karten.elementAt(i)[j]) {
						case 0: text+=("Reiter, "); break;
						case 1: text+=("Kanonen, "); break;
						case 2: text+=("Soldaten, "); break;
						case 3: text+=("Joker, "); break;
					}
				}
				options.add(text);
			}
			String input="";
			while(input.equals("")) {
				Object[] possibilities = options.toArray();
				input = (String) JOptionPane.showInputDialog(
						Fenster.frame,
						"Folgende Karten eintauschen:",
						"Karten eintauschen:",
						JOptionPane.QUESTION_MESSAGE,
						null,
						possibilities,
						possibilities[0]);
				input = (input==null) ? "" : input; 
			}
			int kartenSatzNr = options.indexOf(input);
			if(!pflicht&&kartenSatzNr>0) {
				kartenSatzNr--;
				risiko.kartenEinloesen(karten.elementAt(kartenSatzNr));
			}
			if(pflicht) {
				risiko.kartenEinloesen(karten.elementAt(kartenSatzNr));
				risiko.setPhase(60);
				this.spielerMenu();
			}
		}
	}
	
	/**
	 * Ldt den letzten gespiecherten Spielstand <i>(Spielphase: 7)</i>
	 */
	private void laden() {
		this.addLogEintrag(risiko.laden()); 
		
		Vector<Land> laender = new Vector<Land>(risiko.getAllLaender());
		WeltPanel.setLaender(laender);
		WeltPanel.zeichneLaender();
		
		this.beendeZug(false);
	}
		
	/**
	 * Zeigt Verteidigungspanel an <i>(Spielphase: 50)</i>
	 */
	private void verteidige() {
		Land angriffsland = risiko.getAngreifer();
		Land verteidigungsland = risiko.getVerteidiger();
		int angreifer = risiko.getEinheitenAngriff();
		
		JPanel menuPanel = (JPanel) gui.wichtigeKomponenten.get("MenuPanel");
		CardLayout cards = (CardLayout) menuPanel.getLayout();
		cards.show(menuPanel, "Verteidigung");
		
		MenuPanel.ursprung.get("Verteidigung").setText(angriffsland.toString());
		MenuPanel.ursprung.get("Angreifer").setText(angreifer+"");
		MenuPanel.ziel.get("Verteidigung").setText(verteidigungsland.toString());
		
		int max = (verteidigungsland.getAnzEinheiten()>2) ? 2 : 1;
		MenuPanel.anzahl.get("Verteidigung").setModel(new SpinnerNumberModel(1,1,max,1));
		
		this.addLogEintrag(verteidigungsland.getBesitzer()+" muss verteidigen");
	}
	
	/**
	 * Fhrt Verteidigung aus <i>(Spielphase: 50)</i>
	 */
	private void starteVerteidigung() {
		
		int verteidiger = (Integer) MenuPanel.anzahl.get("Verteidigung").getValue();

		try {
			//Sind die Eingaben korrekt, wird verteidigt und die Zahlen werden ausgedruckt
			risiko.verteidige(verteidiger);
			
			Vector<Integer> angreiferWuerfel = risiko.getWuerfelAngreifer();
			for(int i=0; i<angreiferWuerfel.size(); i++) {
				WuerfelPanel.setZahl(i,angreiferWuerfel.elementAt(i));
			}
			if(angreiferWuerfel.size()<3) {
				for(int i=angreiferWuerfel.size(); i<3; i++) {
					WuerfelPanel.deaktiviere(i);
				}
			}
			Vector<Integer> verteidigerWuerfel = risiko.getWuerfelVerteidiger();
			for(int i=0; i<verteidigerWuerfel.size(); i++) {
				WuerfelPanel.setZahl(i+3,verteidigerWuerfel.elementAt(i));
			}
			if(verteidigerWuerfel.size()<2) {
				WuerfelPanel.deaktiviere(4);
			}
			risiko.weiter();
		} catch (NichtGenugWuerfelException e) {
			this.setFehlerEintrag(e);
		} catch (VerteidigerUebersteigtAngreiferException e) {
			this.setFehlerEintrag(e);
		} catch (NichtGenugEinheitenException e) {
			this.setFehlerEintrag(e);
		} finally {
			this.spielerMenu();
		}
	}
	
	/**
	 * Zeigt Verstrkungspanel an <i>(Spielphase: 60)</i>
	 */
	private void verstaerkung() {
		int anzEinheiten = risiko.getVerstaerkung();
		this.reset();
		JPanel menuPanel = (JPanel) gui.wichtigeKomponenten.get("MenuPanel");
		CardLayout cards = (CardLayout) menuPanel.getLayout();
		cards.show(menuPanel, "Verstaerkung");
		
		MenuPanel.ursprung.get("Verstaerkung").setText(anzEinheiten+"");
		MenuPanel.starten.get("Verstaerkung").setEnabled(false);
		klickStatus = 2;
	}
	
	/**
	 * Lsst einen Spieler eine Zusatzeinheit verteilen <i>(Spielphase: 60)</i>
	 */
	private void starteVerstaerkung() {
		int rest = Integer.parseInt(MenuPanel.ursprung.get("Verstaerkung").getText());
		
		risiko.setzeEinheiten(ziel,1);
		rest--;
		MenuPanel.ursprung.get("Verstaerkung").setText(rest+"");
		
		if(rest==0) {
			risiko.weiter();
			this.spielerMenu();
		}
	}
	
	/**
	 * Lsst nach einem gewonnenen Land Einheiten nachrcken <i>(Spielphase: 70)</i>
	 */
	private void nachruecken() {
		WeltPanel.aktualisiere();
		JPanel menuPanel = (JPanel) gui.wichtigeKomponenten.get("MenuPanel");
		CardLayout cards = (CardLayout) menuPanel.getLayout();
		cards.show(menuPanel,"Transfer");
		
		ursprung = risiko.getAngreifer();
		ziel = risiko.getVerteidiger();
		
		MenuPanel.ursprung.get("Transfer").setText(ursprung.toString());
		MenuPanel.ziel.get("Transfer").setText(ziel.toString());
		
		WeltPanel.highlighteVoll(ursprung);
		WeltPanel.highlighte(ziel);
		
		JSpinner anzahl = (JSpinner) MenuPanel.anzahl.get("Transfer");
		anzahl.setModel(new SpinnerNumberModel(0,0,ursprung.getAnzEinheiten()-1,1));
		
		MenuPanel.starten.get("Transfer").setEnabled(true);
	}
	
	/**
	 * Setzt alle Panels in den Ausgangszustand zurck 
	 * und resettet den Zustand der beiden Lnder-Variablen
	 */
	private void reset() {
		switch(risiko.getPhase()) {
			case 1:  this.setUrsprung(null,true);
					 this.setZiel(null,true);
					 break;
			case 2:  this.setUrsprung(null,true);
					 this.setZiel(null,true);
					 break;
			case 60: this.setZiel(null,true);
					 break;
		}
		this.ursprung = null;
		this.ziel = null;
		this.klickStatus = 0;	
	}
	
	/**
	 * Zeigt ein leeres Panel im MenPanel an
	 */
	private void warten() {
		JPanel menuPanel = (JPanel) gui.wichtigeKomponenten.get("MenuPanel");
		CardLayout cards = (CardLayout) menuPanel.getLayout();
		cards.show(menuPanel,"Leer");
	}

	/**
	 * Prft auf Maus-Klick-Events
	 */
	@Override
	public void mouseClicked(MouseEvent me) {
		
		// Weltkarte als Auslser
		if(me.getComponent() instanceof Hintergrundbild) {
			Hintergrundbild quelle = (Hintergrundbild) me.getComponent(); //Weltkarte
			Color c = new Color(quelle.getRGB(me.getX(), me.getY())); //Farbe des Pixel
			if(!c.equals(Color.BLACK)) { //schwarz = transparent 
				for(Land l : risiko.getAllLaender()) {	
					// Farbe gleich Landesfarbe...
					if(c.equals(l.getFarbe())) {
						this.setInfoEintrag(l.getInfos());
						if(klickStatus==0) { // Status 0: Bereitschaft
							// Nur Infos ausgeben
						} else if(klickStatus==1) { // Status 1: Ursprung gewhlt
							this.setUrsprung(l,false);
						} else if(klickStatus==2) { // Status 2: Ziel gewhlt
							this.setZiel(l,false); 
						}
					}
				}
			}
		}
		
		// Auf die Karten geklickt
		else if(me.getComponent().equals(KartenPanel.karten)) {
			kartenEinloesen(false);
		}
	}

	/**
	 * Prft auf Button-Klick-Events
	 */
	@Override
	public void actionPerformed(ActionEvent ae) {
		if(ae.getActionCommand().equals("Angriffspanel")) {
			risiko.setPhase(1);
			this.spielerMenu();
		} else if(ae.getActionCommand().equals("Transferpanel")) {
			risiko.setPhase(2);
			this.spielerMenu();
		} else if(ae.getActionCommand().equals("Beenden")) {
			risiko.setPhase(3);
			this.spielerMenu();
		} else if(ae.getActionCommand().equals("Angriff")) {
			this.starteAngriff();
		} else if(ae.getActionCommand().equals("Transfer")) {
			this.starteTransfer();
		} else if(ae.getActionCommand().equals("Speichern")) {
			risiko.setPhase(6);
			this.spielerMenu();
		} else if(ae.getActionCommand().equals("Laden")) {
			risiko.setPhase(7);
			this.spielerMenu();
		}else if(ae.getActionCommand().equals("Verteidigung")) {
			this.starteVerteidigung();
		} else if(ae.getActionCommand().equals("Verstaerkung")) {
			this.starteVerstaerkung();
		} else if(ae.getActionCommand().equals("Abbrechen")) {
			this.reset(); risiko.setPhase(0); this.spielerMenu();
		} 
	}
	
	/**
	 * Setzt ein Ursprungsland in einem Panel
	 * 
	 * @param l Land das als Ursprungsland gesetzt werden soll
	 * @param reset Wenn der Zustand des Panels resettet werden soll 
	 * @return true, falls erfolgreich
	 */
	private boolean setUrsprung(Land l, boolean reset) {
		String panel = "";
		switch(risiko.getPhase()) {
			case 1: panel = "Angriff"; break;
			case 2: panel = "Transfer"; break;
		}
		JTextField ursprungsland = (JTextField) MenuPanel.ursprung.get(panel);
		if(reset) {
			this.ursprung = null;
			ursprungsland.setText("");
			this.klickStatus = 0;
		} else {
			this.ursprung = l;
			ursprungsland.setText(ursprung.toString());
			
			if(ursprung.getBesitzer().equals(risiko.getAktiverSpieler())) {
				WeltPanel.highlighteVoll(ursprung);
			} else {
				return false;
			}
			
			if(risiko.getPhase()==1) {
				int max = (ursprung.getAnzEinheiten()>3) ? 3 : ursprung.getAnzEinheiten()-1;
				max = (max<1) ? 1 : max;
				JSpinner anzahl = (JSpinner) MenuPanel.anzahl.get(panel);
				anzahl.setModel(new SpinnerNumberModel(1,1,max,1));
			} else if(risiko.getPhase()==2) {
				JSpinner anzahl = (JSpinner) MenuPanel.anzahl.get(panel);
				anzahl.setModel(new SpinnerNumberModel(0,0,ursprung.getAnzEinheiten()-1,1));
			}
			this.addLogEintrag("Whle das Zielland:");
			klickStatus = 2;
		}
		return true;
	}
	
	/**
	 * Setzt ein Zielland in einem Panel
	 * 
	 * @param l Land das als Zielland gesetzt werden soll
	 * @param reset Wenn der Zustand des Panels resettet werden soll 
	 * @return true, falls erfolgreich
	 */
	private boolean setZiel(Land l,boolean reset) {
		String panel = "";
		switch(risiko.getPhase()) {
			case 1: panel = "Angriff"; break;
			case 2: panel = "Transfer"; break;
			case 60: panel = "Verstaerkung"; break;
		}
		JTextField zielland = (JTextField) MenuPanel.ziel.get(panel);
		if(reset) {
			this.ziel = null;
			this.klickStatus = 0;
			zielland.setText("");
		} else {
			this.ziel = l;
			
			if(risiko.getPhase()==1) {
				if(risiko.isNachbar(ursprung, ziel)
						&&!ziel.getBesitzer().equals(risiko.getAktiverSpieler())) {
					WeltPanel.highlighte(ziel);
				} else {
					return false;
				}
			} else if((risiko.getPhase()==2)||(risiko.getPhase()==60)) {
				if(ziel.getBesitzer().equals(risiko.getAktiverSpieler())) {
					if(risiko.getPhase()==60) WeltPanel.aktualisiere();
					WeltPanel.highlighte(ziel);
				} else {
					return false;
				}
			}
			
			zielland.setText(ziel.toString());
			MenuPanel.starten.get(panel).setEnabled(true);
		}
		return true;
	}
	
	/**
	 * Fgt einen Eintrag in die Historie hinzu
	 * 
	 * @param text der hinzugefgt werden soll
	 */
	private void addLogEintrag(String text) {
		JTextArea ausgabe = (JTextArea) gui.wichtigeKomponenten.get("Historie");
		for(int anz=0; anz<2; anz++) {
			for(int i=0; i<ausgabe.getColumns(); i++) {
				ausgabe.append("+");
			}
			ausgabe.append("\n");
		}
		this.logischeTextEingabe(text, ausgabe);
		ausgabe.append("\n");
	}
	
	/**
	 * Fgt einen Eintrag ins Info-Feld, falls noch Platz ist
	 * 
	 * @param text der hinzugefgt werden soll
	 */
	private void setInfoEintrag(String text) {
		JTextArea ausgabe = (JTextArea) gui.wichtigeKomponenten.get("Information");
		if((ausgabe.getText().length()+text.length())>=(ausgabe.getColumns()*10)) ausgabe.setText("");
		for(int anz=0; anz<2; anz++) {
			for(int i=0; i<ausgabe.getColumns(); i++) {
				ausgabe.append("+");
			}
			ausgabe.append("\n");
		}
		this.logischeTextEingabe(text, ausgabe);
		ausgabe.append("\n");
	}
	
	/**
	 * Gibt eine Fehlermeldung im Fehler-Feld aus 
	 * 
	 * @param e auftretender Fehler
	 */
	private void setFehlerEintrag(Exception e) {
		JTextArea ausgabe = (JTextArea) gui.wichtigeKomponenten.get("Fehler");
		ausgabe.setText("");
		for(int anz=0; anz<2; anz++) {
			for(int i=0; i<ausgabe.getColumns(); i++) {
				ausgabe.append("+");
			}
			ausgabe.append("\n");
		}
		this.logischeTextEingabe(e.getMessage(), ausgabe);
		ausgabe.append("\n");
	}
	
	/**
	 * Formatiert den Text logisch
	 * 
	 * @param text auszugebender Text
	 * @param tArea TextArea in der der text ausgegeben wird
	 */
	private void logischeTextEingabe(String text, JTextArea tArea) {
		String[] zeilen = text.split("\n");
		for(String zeile : zeilen) {
			String[] woerter = zeile.split(" ");
			int restzeichen = tArea.getColumns();
			for(String wort : woerter) {
				if(wort.length()<restzeichen) {
					tArea.append(wort+" ");
					restzeichen -= wort.length()+1;
				} else {
					tArea.append("\n");
					restzeichen = tArea.getColumns();
					tArea.append(wort+" ");
					restzeichen -= wort.length()+1;
				}
			}
			tArea.append("\n");
		}
	}

}

