package risiko.local.ui.cui;

import java.awt.Button;
import java.awt.Color;
import java.awt.TextArea;
import java.util.Vector;

import risiko.local.domain.SpielMgr;
import risiko.local.domain.exceptions.KeineNachbarnException;
import risiko.local.domain.exceptions.LandBeteiligtException;
import risiko.local.domain.exceptions.LandNichtImBesitzException;
import risiko.local.domain.exceptions.NichtBenachbartException;
import risiko.local.domain.exceptions.NichtGenugEinheitenException;
import risiko.local.domain.exceptions.NichtGenugWuerfelException;
import risiko.local.domain.exceptions.VerteidigerUebersteigtAngreiferException;
import risiko.local.valueobjects.Land;

/**
 * Kommandezeilen-basierte Benutzeroberflche als eigenstndiger Thread
 * 
 * @author dgrosche
 * @version 2009-05-31
 *
 */
public class CUI extends Thread {
	
	/**
	 * Hauptklasse (Ablaufplan des Threads)
	 */
	public synchronized void run() {
		println("Herzlich Willkommen beim Konsolen-Risiko");
		println("****************************************");
		println("");
		println("[1] Neues Spiel starten");
		println("[2] Spiel laden");
		println("");
		int auswahl = 0;
		print("Auswahl: ");
		while(!((auswahl>0)&&(auswahl<3))) {
			try {
				auswahl = Integer.parseInt(einlesen());
			} catch(NumberFormatException e1) {
				auswahl = 0;
				printError(e1);
			}
		}
		println("");
		if(auswahl==1) neuesSpiel();
		if(auswahl==2) risiko.laden();
		while(risiko.getPhase()!=100) {
			this.spielerMenu();
		}
		if(this.risiko.isGewonnen()) {
			println("****************************************");
			println("****************************************");
			println("****************************************");
			println("");
			println("Herzlichen Glckwunsch, "+this.risiko.getAktiverSpieler()+"!");
			println("");
			println("Du hast gewonnen!");
			println("");
			this.printLaender();
		}
		
	}
	
	/**
	 * Gelesene Eingabe, wird von Konsole berschrieben
	 */
	public String in;
	
	/**
	 * Ausgabe-Feld
	 */
	public TextArea out;
	
	/**
	 * Senden-Button bermittelt Eingabe von Konsole
	 */
	public Button senden;
	
	/**
	 * Gibt an ob der Programmablauf fortgefahren werden kann
	 */
	public boolean weiter;
	
	private SpielMgr risiko;
	
	/**
	 * Konstruktor mit Hauptverwaltungsklasse 
	 * 
	 * @param out Ausgabekonsole
	 * @param senden Button der Eingabe bermittelt
	 */
	public CUI(TextArea out, Button senden) {
		this.in = "";
		this.risiko = new SpielMgr();
		this.out = out;
		this.senden = senden;
	}
	
	/**
	 * Unterbricht den Thread und wartet auf eine Eingabe
	 * 
	 * @return Eingabestring
	 */
	public String einlesen() {
		senden.setEnabled(true);
		
		try {
			this.wait();
		} catch(InterruptedException e) {
		}
		if(weiter) {
			this.println(in);
			weiter = false;
			senden.setEnabled(false);
		}
		return in;
	}
	
	/**
	 * Fgt Konsole Text und Zeilenumbruch hinzu 
	 * 
	 * @param text auszugebender Text
	 */
	public void println(String text) {
		out.setForeground(Color.BLACK);
		out.append(text+"\n");
		out.repaint();
	}
	
	/**
	 * Fgt Konsole Text ohne Zeilenumbruch hinzu
	 * 
	 * @param text auszugebender Text
	 */
	public void print(String text) {
		out.setForeground(Color.BLACK);
		out.append(text);
		out.repaint();
	}
	
	/**
	 * Fgt Konsole Fehlermeldung hinzu und ndert Schriftfarbe auf Rot
	 * 
	 * @param e auszugebender Fehler
	 */
	public void printError(Exception e) {
		out.setForeground(Color.RED);
		out.append(e.toString()+"\n");
		out.repaint();
	}
	
	/*
	 * Neues Spiel erstellen
	 */
	
	/**
	 * Initialisiert das Spiel, d.h.
	 * <ul>
	 * <li>		Spieler werden registriert			</li>
	 * <li>		Lnder werden verteilt				</li>
	 * <li>		Startreihenfolge wird festgelegt	</li>
	 * <li>		Anfangseinheiten werden gesetzt		</li>
	 * </ul>
	 */
	public void neuesSpiel() {
		//Anzahl der Spieler wird bestimmt. Die Zahl muss zwischen 1 und 6 liegen
		print("Anzahl der Spieler: ");
		int anzSpieler = 0;
		while(!(anzSpieler>2 && anzSpieler<7)) {
			try {
				anzSpieler = Integer.parseInt(einlesen());
			} catch (NumberFormatException e) {
				anzSpieler = 0;
				printError(e);
			}
		}
		
		//Spieler werden hinzugefgt
		for(int i=1; i<=anzSpieler; i++) {
			println("");
			println(i+". Spieler");
			println("++++++++++");
			this.addSpieler();
		}
		println("****************************************");
		
		//Spieler erhalten Lnder und Missioen
		risiko.verteileLaender();
		risiko.verteileMissionen();
		
		//Missionen oder Welt erobern
		println("[1] Missionen ausfhren");
		println("[2] Welt erobern");
		println("");
		int auswahl = 0;
		print("Auswahl: ");
		while(!((auswahl>0)&&(auswahl<3))) {
			try {
				auswahl = Integer.parseInt(einlesen());
			} catch(NumberFormatException e1) {
				auswahl = 0;
				printError(e1);
			}
		}
		if(auswahl==2) risiko.weltErobern();
		println("****************************************");
		
		//Die Startreihenfolge wird ausgewrfelt
		Vector<Integer> zahlen = risiko.reihenfolge();
		println("Fr die Startreihenfolge wurden folgende Zahlen gewrfelt:");
		for(int i=0; i<zahlen.size(); i++) {
			println("   "+risiko.getAllSpieler().elementAt(i)+": "+zahlen.elementAt(i));
			risiko.beendeZug();
		}
		
		//Der erste Spieler beginnt mit dem Setzen von zustzlichen Einheiten
		risiko.setPhase(60);		
	}
	
	/**
	 * Fragt die Spielernamen ab und fgt sie hinzu.
	 */
	public void addSpieler() {
		print("Name des Spielers: ");
		String name = "";
		while (name.equals("")) {
			try {
				name = einlesen();
			} catch (Exception e) {
				name = "";
				printError(e);
			}
		}
		risiko.addSpieler(name);
	}
	
	/*
	 * Menfhrung
	 */
	
	/**
	 * Gibt das Spielphasen entsprechende Men aus:
	 * <br>
	 *   <br><b> Phase 0:   </b> Hauptmen					
	 *   <br><b> Phase 1:   </b> Angriffsmen				
	 *   <br><b> Phase 2:   </b> Verschiebenmen			
	 *   <br><b> Phase 3:   </b> Lnder drucken			
	 *   <br><b> Phase 4:   </b> Zug beenden				
	 *   <br><b> Phase 5:   </b> Karten einlsen			
	 *   <br><b> Phase 6:   </b> Spiel speichern			
	 *   <br><b> Phase 7:   </b> Spiel laden				
	 *   <br><b> Phase 50:  </b> Verteidigen				
	 *   <br><b> Phase 60:  </b> Verstrkung setzen		
	 *   <br><b> Phase 70:  </b> Einheiten nachrcken		
	 *   <br><b> Phase 100: </b> Spiel beenden			
	 *   
	 */
	public void spielerMenu() {
		switch (risiko.getPhase()) {
			case 0: hauptmenu(); break;
			case 1: neuerAngriff(); break;
			case 2: verschieben(); break;
			case 3: risiko.weiter(); break;
			case 4: printLaender(); break; 
			case 5: kartenEinloesen(); break;
			case 6: speichern(); break;
			case 7: laden(); break;
			case 50: verteidige(); break;
			case 60: verstaerkung(); break;
			case 70: nachruecken(); break;
		}
	}
	
	/**
	 * Gibt das Hauptmen eines Spielers aus <i>(Spielphase: 0)</i>
	 */
	public void hauptmenu() {
		println("");
		println("****************************************");
		println(risiko.getAktiverSpieler()+", du hast folgende Mglichkeiten:");
		println("");
		println("MISSION: "+risiko.getAktiverSpieler().getMission());
		println("");
		println("   [1] Angriff starten");
		println("   [2] Einheiten verschieben");
		println("   [3] Zug beenden");
		println("");
		println("   [4] Eigene Lnder anzeigen");
		println("   [5] Karten eintauschen");
		println("");
		println("   [6] Spiel speichern");
		println("   [7] Spiel laden");
		println("   [8] Spiel beenden");
		println("");
		print("Auswahl: ");
		int auswahl = 0;
		while(!(auswahl>0 && auswahl<=8)) {
			try {
				auswahl = Integer.parseInt(einlesen());
			} catch (NumberFormatException e) {
				auswahl = 0;
				printError(e);
			}
		}
		println("****************************************");
		println("");
		
		//Die gewhlte Zahl wird direkt als nchste Phase weitergegeben
		if(auswahl==8) auswahl = 100; //... nur bei Spiel beenden nicht
		risiko.setPhase(auswahl);
	}
	
	/**
	 * Startet neuen Kampf <i>(Spielphase: 1)</i>
	 */
	public void neuerAngriff() {
		println("Von wo willst du angreifen?");
		this.printLaender();
		print("Nummer eingeben: ");
		int landNr = 0;
		while(!(landNr>0 && landNr<=risiko.getLaender(risiko.getAktiverSpieler()).size())) {
			try {
				landNr = Integer.parseInt(einlesen());
			} catch (NumberFormatException e) {
				landNr = 0;
				printError(e);
			} 
		}
		landNr--;
		Land angriffsland = risiko.getLaender(risiko.getAktiverSpieler()).elementAt(landNr);
		try {
			Vector<Land> fremdeNachbarn = risiko.getFremdeNachbarn(angriffsland);
			println("");
			println("Welches Land soll angegriffen werden?");
			int i = 1;
			for(Land l : fremdeNachbarn) {
				println("   "+(i)+")"+l+" ["+l.getBesitzer()+"]");
				i++;
			}
			print("Nummer eingeben: ");
			landNr = 0;
			while(!(landNr>0 && landNr<=fremdeNachbarn.size())) {
				try {
					landNr = Integer.parseInt(einlesen());
				} catch (NumberFormatException e) {
					landNr = 0;
					printError(e);
				} 
			}
			landNr--;
			Land verteidigungsland = fremdeNachbarn.elementAt(landNr);

			print("Mit wievielen Einheiten soll angegriffen werden? ");
			int angreifer = 0;
			while(!(angreifer>0)) {
				try {
					angreifer = Integer.parseInt(einlesen());
				} catch (NumberFormatException e) {
					angreifer = 0;
					printError(e);
				}
			}
			
			//Sind die Eingaben formal richtig, wird ein Angriff gestartet
			risiko.starteAngriff(angriffsland, verteidigungsland, angreifer);
			risiko.weiter();
		} catch (LandNichtImBesitzException e) {
			printError(e);
		} catch (KeineNachbarnException e) {
			printError(e);
			risiko.setPhase(0); //Es kann nicht angegriffen werden, also Hauptmen 
		} catch (NichtBenachbartException e) {
			printError(e);
			risiko.setPhase(0); //Es kann nicht angegriffen werden, also Hauptmen
		} catch (NichtGenugEinheitenException e) {
			printError(e);
			risiko.setPhase(0); //Es kann nicht angegriffen werden, also Hauptmen
		} catch (NichtGenugWuerfelException e) {
			printError(e);
		}
	}
	
	/**
	 * Verschiebt Einheiten <i>(Spielphase: 2)</i>
	 */
	public void verschieben() {
		println("Verschieben aus Land: ");
		this.printLaender();
		print("Nummer eingeben: ");
		int landNr = 0;
		while(!(landNr>0 && landNr<=risiko.getLaender(risiko.getAktiverSpieler()).size())) {
			try {
				landNr = Integer.parseInt(einlesen());
			} catch (NumberFormatException e) {
				landNr = 0;
				printError(e);
			}
		}
		landNr--;
		Land ursprungsland = risiko.getLaender(risiko.getAktiverSpieler()).elementAt(landNr);
		println("");
		try {
			int i=1;
			for(Land l : risiko.getEigeneNachbarn(ursprungsland)) {
				println("   "+(i)+")"+l);
				i++;
			}
			print("Nummer eingeben: ");
			landNr = 0;
			while(!(landNr>0 && landNr<=risiko.getEigeneNachbarn(ursprungsland).size())) {
				try {
					landNr = Integer.parseInt(einlesen());
				} catch (NumberFormatException e) {
					landNr = 0;
					printError(e);
				}
			}
			landNr--;
			Land zielland = risiko.getEigeneNachbarn(ursprungsland).elementAt(landNr);
			print("Wie viele Einheiten sollen verschoben werden? ");
			int anzEinheiten = -1; 
			while(!(anzEinheiten>=0 && anzEinheiten<ursprungsland.getAnzEinheiten())) {
				try {
					anzEinheiten = Integer.parseInt(einlesen());
				} catch (NumberFormatException e) {
					anzEinheiten = 0;
					printError(e);
				}
			}
			
			//Sind die Eingaben formal richtig, werden die Einheiten verschoben
			risiko.verschiebeEinheiten(ursprungsland, zielland, anzEinheiten);
			println("");
			
			//Das Verschieben wird solange wiederholt, wie man will
			print("Nochmal verschieben? (j/n): ");
			char nochmal ='0';
			while(!((nochmal=='j')||(nochmal=='J')||(nochmal=='n')||(nochmal=='N'))) {
				try {
					nochmal = einlesen().charAt(0);
				} catch (Exception e) {
					nochmal = '0';
					printError(e);
				}
			}
			
			//Wird nein ausgwhlt, geht's zur nchsten Phase weiter
			if((nochmal=='n')||(nochmal=='N')) risiko.weiter();
			
		} catch (NichtGenugEinheitenException e) {
			printError(e);
		} catch (KeineNachbarnException e) {
			printError(e);
		} catch (LandBeteiligtException e) {			
			printError(e);
		} catch (LandNichtImBesitzException e) {
			printError(e);
		} catch (NichtBenachbartException e) {
			printError(e);
		}		
	}
	
	/**
	 * Gibt alle Lnder des aktiven Spieler aus <i>(Spielphase: 3)</i>
	 */
	public void printLaender() {
		println(risiko.getAktiverSpieler()+", deine Lnder sind:");
		for(int i=0; i<risiko.getLaender(risiko.getAktiverSpieler()).size(); i++) {
			println("   "+(i+1)+") "+risiko.getLaender(risiko.getAktiverSpieler()).elementAt(i));
		}
		println("");
		
		//Wenn die Methode als Spielphase aufgerufen wird, muss die Phase anschlieend verndert werden.
		if(risiko.getPhase()==4) risiko.weiter();
	}
	
	/**
	 * <br>Gibt Karten zurck. 
	 * <br>Prft ob einlsbare Karten vorhanden sind.
	 * <br>Lsst Karten einlsen.
	 * <br><i>(Spielphase: 5)</i> 
	 */
	public void kartenEinloesen() {
		boolean pflicht = (risiko.getAktiverSpieler().getKarten().size()>4) ? true : false;
		println(risiko.getAktiverSpieler()+", du hast folgende Karten:");
		for(int i : risiko.getAktiverSpieler().getKarten()) {
			//Die Karten sind eigentlich nur Zahlen, entsprechend dieser Liste
			switch(i) {
				case 0: println("   Reiter"); break;
				case 1: println("   Kanonen"); break;
				case 2: println("   Soldaten"); break;
				case 3: println("   Joker"); break;
				default: print(""); break;
			}
		}
		println("");
		if(!risiko.pruefKarten()) {
			println("Du kannst keine Karten eintauschen!");
			risiko.weiter();
		} else {
			println("Du kannst Karten eintauschen!");
			Vector<int[]> karten = risiko.getEinloesbareKarten(); 
			if(!pflicht) {
				println("   0) keine Karten eintauschen!");
			}
			for(int i=0; i<karten.size(); i++) {
				print("   "+(i+1)+") ");
				for(int j=0; j<3; j++) {
					switch(karten.elementAt(i)[j]) {
						case 0: print("Reiter, "); break;
						case 1: print("Kanonen, "); break;
						case 2: print("Soldaten, "); break;
						case 3: print("Joker, "); break;
					}
				}
				println("");
			}
			println("");
			print("Nummer eingeben: ");
			int min = pflicht ? 0 : -1;
			int kartSatzNr = min;
			while(!(kartSatzNr>min && kartSatzNr<=karten.size())) {
				try {
					kartSatzNr = Integer.parseInt(einlesen());
				} catch (NumberFormatException e) {
					kartSatzNr = min;
					printError(e);
				}
			}
			if(kartSatzNr>0) {
				kartSatzNr--;
				risiko.kartenEinloesen(karten.elementAt(kartSatzNr));
			}
			if(pflicht) {
				risiko.setPhase(60);
			} else {
				risiko.weiter();
			}
		}		
	}
	
	/**
	 * Speichert einen neuen Spielstand <i>(Spielphase: 6)</i>
	 */
	public void speichern() {
		println("****************************************");
		println(risiko.speichern());
		println("****************************************");
		risiko.weiter();
	}
	
	/**
	 * Ldt den letzten vorhandenen Spielstand <i>(Spielphase: 7)</i>
	 */
	public void laden() {
		println("****************************************");
		println(risiko.laden());
		println("****************************************");
		risiko.weiter();
	}
	
	/**
	 * Fhrt Verteidigung aus <i>(Spielphase: 50)</i>
	 */
	public void verteidige() {
		println("****************************************");
		Land verteidigungsland = risiko.getVerteidiger();
		print(verteidigungsland.getBesitzer()+", mit wievielen Einheiten wollen Sie verteidigen? ");
		int verteidiger = 0;
			
		while(!(verteidiger>0)) {
			try {
				verteidiger = Integer.parseInt(einlesen());
			} catch (NumberFormatException e) {
				verteidiger = 0;
				printError(e);
			}
		}
		println("****************************************");
		String verteidigername = verteidigungsland.getBesitzer().getName();
		Land angriffsland = risiko.getAngreifer();
		try {
			//Sind die Eingaben korrekt, wird verteidigt und die Zahlen werden ausgedruckt
			risiko.verteidige(verteidiger);
			println(angriffsland.getBesitzer()+", deine Wrfelzahlen:");
			for(int zahl : risiko.getWuerfelAngreifer()) {
				println("   "+zahl);
			}
			println("");
			println(verteidigername+", deine Wrfelzahlen:");
			for(int zahl : risiko.getWuerfelVerteidiger()) {
				println("   "+zahl);
			}
			println("****************************************");
			risiko.weiter();
		} catch (NichtGenugWuerfelException e) {
			printError(e);
			//Wenn der Fehler vom Angreifer ausgeht, gehts zum Hauptmen
			if(e.behandle()) risiko.weiter();
		} catch (VerteidigerUebersteigtAngreiferException e) {
			printError(e);
		} catch (NichtGenugEinheitenException e) {
			printError(e);
			//Wenn der Fehler vom Angreifer ausgeht, gehts zum Hauptmen
			if(e.behandle()) risiko.weiter();
		}
	}
	
	/**
	 * Lsst einen Spieler Zusatzeinheiten verteilen <i>(Spielphase: 60)</i>
	 */
	public void verstaerkung() {
		println("****************************************");
		println("");
		println("MISSION: "+risiko.getAktiverSpieler().getMission());
		println("");
		println(risiko.getAktiverSpieler()+", du kannst "+risiko.getVerstaerkung()+" zustzliche Einheiten verteilen:");
		this.printLaender();
		for(int i=0; i<risiko.getVerstaerkung(); i++) {
			print("Einheiten hinzufgen zu Land Nummer: ");
			int landNr = 0;
			while(!(landNr>0 && landNr<=risiko.getLaender(risiko.getAktiverSpieler()).size())) {
				try {
					landNr = Integer.parseInt(einlesen());
				} catch (NumberFormatException e) {
					landNr = 0;
					printError(e);
				}
			}
			landNr--;
			risiko.setzeEinheiten(risiko.getLaender(risiko.getAktiverSpieler()).elementAt(landNr),1);
		}
		risiko.weiter();
	}
	
	/**
	 * Lsst nach einem gewonnenen Land Einheiten nachrcken <i>(Spielphase: 70)</i>
	 */
	public void nachruecken() {
		print("Wie viele Einheiten sollen nachrcken? ");
		int anzEinheiten = -1;
		while(anzEinheiten<0) {
			try {
				anzEinheiten = Integer.parseInt(einlesen());
			} catch (NumberFormatException e) {
				anzEinheiten = -1;
				printError(e);
			}
		}
		try {
			risiko.verschiebeEinheiten(risiko.getAngreifer(), 
									   risiko.getVerteidiger(), 
									   anzEinheiten);
		} catch (NichtGenugEinheitenException e) {
			printError(e);
		} catch (LandBeteiligtException e) {
			printError(e);
		} catch (LandNichtImBesitzException e) {
			printError(e);
		} catch (NichtBenachbartException e) {
			printError(e);
		}
		risiko.weiter();
	}
		


}
