package risiko.local.domain;

import java.awt.Color;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.Serializable;
import java.util.Vector;

import risiko.local.domain.exceptions.NichtGenugEinheitenException;
import risiko.local.persistence.PersMgr;
import risiko.local.valueobjects.Kontinent;
import risiko.local.valueobjects.Land;
import risiko.local.valueobjects.Spieler;

/**
 * Verwaltet Lnder, Kontinente und Lnderbeziehungen.
 * 
 * @author hhollstein
 * @version 2009-06-08
 *
 */
public class WeltMgr implements Serializable {
	
	/**
	 * Serielle Versions-ID
	 */
	private static final long serialVersionUID = 3418786859452062753L;
	
	/*
	 * Eigenschaften:
	 * - Kontinentenliste (Vector)
	 * - Lnderliste (Vector)
	 * - Beziehung zwischen Lndern (benachbarte Lnder) (Map)
	 * 
	 *     1
	 *   /   \
	 *  2 --- 3
	 *         \
	 *          4
	 *             
	 *   1 2 3 4
	 * 1 T T T F
	 * 2 T T T F
	 * 3 T T T T
	 * 4 F F T T
	 * 
	 * boolean[][] beziehung = new [Anzahl der Lnder][Anzahl der Lnder] 
	 * 
	 * 
	 * Methoden:
	 * - setBesitzer(Land l, Spieler spieler, int x) 
	 * 		-> setzt spieler als neuen Besitzer des Landes 
	 * 		   und x Einheiten auf das Land
	 * - berprfen ob ein Kontinent erobert ist (gibt boolean zurck)
	 * - addLandToKontinent(Land, Kontinent);
	 * - verteileLnder -> verteilt die Lnder gleichmug auf die Spieler durch setBesitzer(spieler, 1)
	 * - addEinheiten (Land l, int x) -> fgt x Einheiten zum Land hinzu
	 * - removeEinheiten (Land l, int x) -> entefernt x Einheiten aus dem Land
	 * - verschiebeEinheiten(Land land1, Land land2, int x) -> verschiebt x Einheiten von land1 nach land2.
	 * 
	 * Konstruktor:
	 * - Kontinent- und Lnderlisten erstellen
	 * - Kontinente anlegen und zur Kontinentenliste hinzufgen
	 * - Lnder anlegen und zur Liste hinzufgen
	 * - Lnderbeziehungen werden gesetzt
	 * - Lnderliste der Kontinente werden erstellt
	 * 
	 */
	private Vector<Kontinent> kontinentenListe;
	private Vector<Land> laenderListe;
	private boolean[][] beziehung;
	
	/**
	 * Erstellt Welt aus Inhalt von Dateien
	 * 
	 * @param persistenz Datenspeicherung
	 * @throws FileNotFoundException , wenn Datei nicht vorhanden
	 * @throws IOException , wenn Fehler beim Einlesen
	 */
	public WeltMgr(PersMgr persistenz) throws FileNotFoundException, IOException {
		kontinentenListe = new Vector<Kontinent>();
		laenderListe = new Vector<Land>();
		persistenz.dateiLaden();
		
		Vector<String[]> kontinente = new Vector<String[]>();
		for(String zeile : persistenz.getSubstring('$')) {
			kontinente.add(zeile.split(";"));
		}
		for(String[] teile : kontinente) {
			Kontinent k = new Kontinent(teile[0],Integer.parseInt(teile[1]));
			kontinentenListe.add(k);
		}
		
		for(String zeile : persistenz.getSubstring('#')) {
			String[] teile = zeile.split(";");
			String name = teile[0];
			int farbe = Integer.parseInt(teile[1]);
			Land l = new Land(name,new Color(farbe));
			laenderListe.add(l);
		}
		
		Vector<int[]> kontinentLaender = new Vector<int[]>();
		for(String zeile : persistenz.getSubstring('&')) {
			String[] teile = zeile.split(";");
			int[] erg = new int[teile.length];
			for(int i=0; i<teile.length; i++) {
				erg[i] = Integer.parseInt(teile[i]);
			}
			kontinentLaender.add(erg);
		}
		while(kontinentLaender.size()>0) {
			Vector<Land> einKontinent = new Vector<Land>();
			int j = kontinentLaender.firstElement()[0];
			int anzahl = 0;
			for(int[] zuordnung : kontinentLaender) {
				if (zuordnung[0]==j) {
					einKontinent.add(laenderListe.elementAt(zuordnung[1]));
					anzahl++;
				}
			}
			kontinentenListe.elementAt(j).setLaenderListe(einKontinent);
			for(int zaehler=0; zaehler<anzahl; zaehler++) {
				kontinentLaender.removeElementAt(0);
			}
		}
		
		beziehung = new boolean[laenderListe.size()][laenderListe.size()];
		Vector<int[]> bez = new Vector<int[]>();
		for(String zeile : persistenz.getSubstring('%')) {
			String[] teile = zeile.split(";");
			int[] erg = new int[teile.length];
			for(int i=0; i<erg.length; i++) {
				erg[i] = Integer.parseInt(teile[i]);
			}
			bez.add(erg);
		}
		for(int[] zuordnung : bez) {
			beziehung[zuordnung[0]][zuordnung[1]] = true;
		}
		
		
	
	}
	
	/**
	 * Erstellt Lnder, Kontinente, Lnderbeziehungen
	 */
	public WeltMgr(){
		//KontinentenListe erstellen
		kontinentenListe = new Vector<Kontinent>();
		
		//LnderListe erstellen
		laenderListe = new Vector<Land>();
		
		//Kontinente anlegen und zur kontinentenListe hinzufgen
		kontinentenListe.add(new Kontinent("Nordamerika", 5));	//0
		kontinentenListe.add(new Kontinent("Sdamerika", 2));	//1
		kontinentenListe.add(new Kontinent("Europa", 5));		//2
		kontinentenListe.add(new Kontinent("Afrika", 3));		//3
		kontinentenListe.add(new Kontinent("Asien", 7));		//4
		kontinentenListe.add(new Kontinent("Australien", 2));	//5
		
		//Lnder anlegen und zur laenderListe hinzufgen
		//Part 1------------
		laenderListe.add(new Land("Alaska",new Color(255,0,0)));
		laenderListe.add(new Land("Nordwest-Territorium",new Color(255,255,0)));
		laenderListe.add(new Land("Alberta",new Color(0,255,0)));
		laenderListe.add(new Land("Ontario",new Color(0,255,255)));
		laenderListe.add(new Land("Quebec", new Color(255,0,255)));
		laenderListe.add(new Land("Weststaaten",new Color(96,56,17)));
		laenderListe.add(new Land("Oststaaten",new Color(49,0,74)));
		laenderListe.add(new Land("Mittelamerika",new Color(0,91,127)));
		laenderListe.add(new Land("Grnland",new Color(0,0,255)));
		//------------------
		
		//Lnderliste des Kontinents "Nordamerika" erzeugen und aus laenderListe die entsprechenden Lnder einfgen
		Vector<Land> nordamerika = new Vector<Land>();//Vektor mit Lndern(Vektor<Land>), da jeder Kontinent eine Lnderliste hat
		nordamerika.add(this.laenderListe.elementAt(0));
		nordamerika.add(this.laenderListe.elementAt(1));
		nordamerika.add(this.laenderListe.elementAt(2));
		nordamerika.add(this.laenderListe.elementAt(3));
		nordamerika.add(this.laenderListe.elementAt(4));
		nordamerika.add(this.laenderListe.elementAt(5));
		nordamerika.add(this.laenderListe.elementAt(6));
		nordamerika.add(this.laenderListe.elementAt(7));
		nordamerika.add(this.laenderListe.elementAt(8));
		this.kontinentenListe.elementAt(0).setLaenderListe(nordamerika);
		
		//Part 2------------
		laenderListe.add(new Land("Venezuela",new Color(0,87,37)));
		laenderListe.add(new Land("Peru",new Color(121,0,0)));
		laenderListe.add(new Land("Brasilien",new Color(130,123,0)));
		laenderListe.add(new Land("Argentinien",new Color(158,0,93)));
		//------------------
		
		//Lnderliste des Kontinents "Sdamerika" erzeugen und aus laenderListe die entsprechenden Lnder einfgen
		Vector<Land> suedamerika = new Vector<Land>();
		suedamerika.add(this.laenderListe.elementAt(9));
		suedamerika.add(this.laenderListe.elementAt(10));
		suedamerika.add(this.laenderListe.elementAt(11));
		suedamerika.add(this.laenderListe.elementAt(12));
		this.kontinentenListe.elementAt(1).setLaenderListe(suedamerika);
		
		//Part 3------------
		laenderListe.add(new Land("Island",new Color(26,19,100)));
		laenderListe.add(new Land("Skandinavien",new Color(89,133,38)));
		laenderListe.add(new Land("Ukraine",new Color(171,160,0)));
		laenderListe.add(new Land("Mittel-Europa",new Color(163,98,9)));
		laenderListe.add(new Land("West-Europa",new Color(158,9,12)));
		laenderListe.add(new Land("Grobritannien",new Color(23,123,47)));
		laenderListe.add(new Land("Sd-Europa",new Color(238,19,91)));
		//------------------
		
		//Lnderliste des Kontinents "Europa" erzeugen und aus laenderListe die entsprechenden Lnder einfgen
		Vector<Land> europa = new Vector<Land>();
		europa.add(this.laenderListe.elementAt(13));
		europa.add(this.laenderListe.elementAt(14));
		europa.add(this.laenderListe.elementAt(15));
		europa.add(this.laenderListe.elementAt(16));
		europa.add(this.laenderListe.elementAt(17));
		europa.add(this.laenderListe.elementAt(18));
		europa.add(this.laenderListe.elementAt(19));
		this.kontinentenListe.elementAt(2).setLaenderListe(europa);
		
		//Part 4------------
		laenderListe.add(new Land("Nordwest-Afrika",new Color(146,38,143)));
		laenderListe.add(new Land("gypten",new Color(147,121,190)));
		laenderListe.add(new Land("Ost-Afrika",new Color(46,48,146)));
		laenderListe.add(new Land("Kongo",new Color(0,84,166)));
		laenderListe.add(new Land("Sd-Afrika",new Color(0,114,188)));
		laenderListe.add(new Land("Madagaskar",new Color(163,212,156)));
		//------------------
		
		//Lnderliste des Kontinents "Afrika" erzeugen und aus laenderListe die entsprechenden Lnder einfgen
		Vector<Land> afrika = new Vector<Land>();
		afrika.add(this.laenderListe.elementAt(20));
		afrika.add(this.laenderListe.elementAt(21));
		afrika.add(this.laenderListe.elementAt(22));
		afrika.add(this.laenderListe.elementAt(23));
		afrika.add(this.laenderListe.elementAt(24));
		afrika.add(this.laenderListe.elementAt(25));
		this.kontinentenListe.elementAt(3).setLaenderListe(afrika);
		
		//Part 5------------
		laenderListe.add(new Land("Ural",new Color(141,199,62)));
		laenderListe.add(new Land("Sibirien",new Color(249,148,28)));
		laenderListe.add(new Land("Jakutien",new Color(238,26,36)));
		laenderListe.add(new Land("Kamtschatka",new Color(168,100,169)));
		laenderListe.add(new Land("Irkutsk",new Color(241,110,170)));
		laenderListe.add(new Land("Mongolei",new Color(96,92,169)));
		laenderListe.add(new Land("China",new Color(60,184,120)));
		laenderListe.add(new Land("Japan",new Color(254,198,137)));
		laenderListe.add(new Land("Siam",new Color(255,246,104)));
		laenderListe.add(new Land("Indien",new Color(124,197,118)));
		laenderListe.add(new Land("Afghanistan",new Color(0,166,81)));
		laenderListe.add(new Land("Mittlerer-Osten",new Color(102,44,145)));
		//------------------
		
		//Lnderliste des Kontinents "Asien" erzeugen und aus laenderListe die entsprechenden Lnder einfgen
		Vector<Land> asien = new Vector<Land>();
		asien.add(this.laenderListe.elementAt(26));
		asien.add(this.laenderListe.elementAt(27));
		asien.add(this.laenderListe.elementAt(28));
		asien.add(this.laenderListe.elementAt(29));
		asien.add(this.laenderListe.elementAt(30));
		asien.add(this.laenderListe.elementAt(31));
		asien.add(this.laenderListe.elementAt(32));
		asien.add(this.laenderListe.elementAt(33));
		asien.add(this.laenderListe.elementAt(34));
		asien.add(this.laenderListe.elementAt(35));
		asien.add(this.laenderListe.elementAt(36));
		asien.add(this.laenderListe.elementAt(37));
		this.kontinentenListe.elementAt(4).setLaenderListe(asien);
		
		//Part 6------------
		laenderListe.add(new Land("Indonesien",new Color(248,142,86)));
		laenderListe.add(new Land("Neuguinea",new Color(245,154,194)));
		laenderListe.add(new Land("Ost-Australien",new Color(125,167,217)));
		laenderListe.add(new Land("West-Australien",new Color(161,134,190)));
		//------------------
		
		//Lnderliste des Kontinents "Australien" erzeugen und aus laenderListe die entsprechenden Lnder einfgen
		Vector<Land> australien = new Vector<Land>();
		australien.add(this.laenderListe.elementAt(38));
		australien.add(this.laenderListe.elementAt(39));
		australien.add(this.laenderListe.elementAt(40));
		australien.add(this.laenderListe.elementAt(41));
		this.kontinentenListe.elementAt(5).setLaenderListe(australien);
		
		
		//Lnderbeziehungen
		beziehung = new boolean[laenderListe.size()][laenderListe.size()];
		
		beziehung[0][0] = true;
		beziehung[0][1] = true;
		beziehung[0][2] = true;
		beziehung[0][29] = true;
		
		beziehung[1][0] = true;
		beziehung[1][1] = true;
		beziehung[1][2] = true;
		beziehung[1][3] = true;
		beziehung[1][8] = true;
		
		beziehung[2][0] = true;
		beziehung[2][1] = true;
		beziehung[2][2] = true;
		beziehung[2][3] = true;
		beziehung[2][5] = true;
		
		beziehung[3][1] = true;
		beziehung[3][2] = true;
		beziehung[3][3] = true;
		beziehung[3][4] = true;
		beziehung[3][5] = true;
		beziehung[3][6] = true;
		beziehung[3][8] = true;
		
		beziehung[4][3] = true;
		beziehung[4][4] = true;
		beziehung[4][6] = true;
		beziehung[4][8] = true;
		
		beziehung[5][2] = true;
		beziehung[5][3] = true;
		beziehung[5][5] = true;
		beziehung[5][6] = true;
		beziehung[5][7] = true;
		
		beziehung[6][3] = true;
		beziehung[6][4] = true;
		beziehung[6][5] = true;
		beziehung[6][6] = true;
		beziehung[6][7] = true;
		
		beziehung[7][5] = true;
		beziehung[7][6] = true;
		beziehung[7][7] = true;
		beziehung[7][9] = true;
		
		beziehung[8][1] = true;
		beziehung[8][3] = true;
		beziehung[8][4] = true;
		beziehung[8][8] = true;
		beziehung[8][13] = true;
		
		beziehung[9][7] = true;
		beziehung[9][9] = true;
		beziehung[9][10] = true;
		beziehung[9][11] = true;
		
		beziehung[10][9] = true;
		beziehung[10][10] = true;
		beziehung[10][11] = true;
		beziehung[10][12] = true;
		
		beziehung[11][9] = true;
		beziehung[11][10] = true;
		beziehung[11][11] = true;
		beziehung[11][12] = true;
		beziehung[11][20] = true;
		
		beziehung[12][10] = true;
		beziehung[12][11] = true;
		beziehung[12][12] = true;
		
		beziehung[13][8] = true;
		beziehung[13][13] = true;
		beziehung[13][14] = true;
		beziehung[13][18] = true;
		
		beziehung[14][13] = true;
		beziehung[14][14] = true;
		beziehung[14][15] = true;
		beziehung[14][16] = true;
		beziehung[14][18] = true;
		
		beziehung[15][14] = true;
		beziehung[15][15] = true;
		beziehung[15][16] = true;
		beziehung[15][19] = true;
		beziehung[15][26] = true;
		beziehung[15][36] = true;
		beziehung[15][37] = true;
		
		beziehung[16][14] = true;
		beziehung[16][15] = true;
		beziehung[16][16] = true;
		beziehung[16][17] = true;
		beziehung[16][18] = true;
		beziehung[16][19] = true;
		
		beziehung[17][16] = true;
		beziehung[17][17] = true;
		beziehung[17][18] = true;
		beziehung[17][19] = true;
		beziehung[17][20] = true;
		
		beziehung[18][13] = true;
		beziehung[18][14] = true;
		beziehung[18][16] = true;
		beziehung[18][17] = true;
		beziehung[18][18] = true;
		
		beziehung[19][15] = true;
		beziehung[19][16] = true;
		beziehung[19][17] = true;
		beziehung[19][19] = true;
		beziehung[19][20] = true;
		beziehung[19][21] = true;
		beziehung[19][37] = true;
		
		beziehung[20][11] = true;
		beziehung[20][17] = true;
		beziehung[20][19] = true;
		beziehung[20][20] = true;
		beziehung[20][21] = true;
		beziehung[20][22] = true;
		beziehung[20][23] = true;
		
		beziehung[21][19] = true;
		beziehung[21][20] = true;
		beziehung[21][21] = true;
		beziehung[21][22] = true;
		beziehung[21][37] = true;
		
		beziehung[22][20] = true;
		beziehung[22][21] = true;
		beziehung[22][22] = true;
		beziehung[22][23] = true;
		beziehung[22][24] = true;
		beziehung[22][25] = true;
		beziehung[22][37] = true;
		
		beziehung[23][20] = true;
		beziehung[23][22] = true;
		beziehung[23][23] = true;
		beziehung[23][24] = true;
		
		beziehung[24][22] = true;
		beziehung[24][23] = true;
		beziehung[24][24] = true;
		beziehung[24][25] = true;
		
		beziehung[25][22] = true;
		beziehung[25][24] = true;
		beziehung[25][25] = true;
		
		beziehung[26][15] = true;
		beziehung[26][26] = true;
		beziehung[26][27] = true;
		beziehung[26][32] = true;
		beziehung[26][36] = true;
		
		beziehung[27][26] = true;
		beziehung[27][27] = true;
		beziehung[27][28] = true;
		beziehung[27][30] = true;
		beziehung[27][31] = true;
		beziehung[27][32] = true;
		
		beziehung[28][27] = true;
		beziehung[28][28] = true;
		beziehung[28][29] = true;
		beziehung[28][30] = true;
		
		beziehung[29][0] = true;
		beziehung[29][28] = true;
		beziehung[29][29] = true;
		beziehung[29][30] = true;
		beziehung[29][31] = true;
		beziehung[29][33] = true;
		
		beziehung[30][27] = true;
		beziehung[30][28] = true;
		beziehung[30][29] = true;
		beziehung[30][30] = true;
		beziehung[30][31] = true;
		
		beziehung[31][27] = true;
		beziehung[31][29] = true;
		beziehung[31][30] = true;
		beziehung[31][31] = true;
		beziehung[31][32] = true;
		beziehung[31][33] = true;
		
		beziehung[32][26] = true;
		beziehung[32][27] = true;
		beziehung[32][31] = true;
		beziehung[32][32] = true;
		beziehung[32][34] = true;
		beziehung[32][35] = true;
		beziehung[32][36] = true;
		
		beziehung[33][29] = true;
		beziehung[33][31] = true;
		beziehung[33][33] = true;
		
		beziehung[34][32] = true;
		beziehung[34][34] = true;
		beziehung[34][35] = true;
		beziehung[34][38] = true;
		
		beziehung[35][32] = true;
		beziehung[35][34] = true;
		beziehung[35][35] = true;
		beziehung[35][36] = true;
		beziehung[35][37] = true;
		
		beziehung[36][15] = true;
		beziehung[36][26] = true;
		beziehung[36][32] = true;
		beziehung[36][35] = true;
		beziehung[36][36] = true;
		beziehung[36][37] = true;
		
		beziehung[37][15] = true;
		beziehung[37][19] = true;
		beziehung[37][21] = true;
		beziehung[37][22] = true;
		beziehung[37][35] = true;
		beziehung[37][36] = true;
		beziehung[37][37] = true;
		
		beziehung[38][34] = true;
		beziehung[38][38] = true;
		beziehung[38][39] = true;
		beziehung[38][41] = true;
		
		beziehung[39][38] = true;
		beziehung[39][39] = true;
		beziehung[39][40] = true;
		beziehung[39][41] = true;
		
		beziehung[40][39] = true;
		beziehung[40][40] = true;
		beziehung[40][41] = true;
		
		beziehung[41][38] = true;
		beziehung[41][39] = true;
		beziehung[41][40] = true;
		beziehung[41][41] = true;
		
	}
	
	/**
	 * -> verschiebt x Einheiten von land1 nach land2.
	 * 
	 * @param land1 Ursprungsland
	 * @param land2 Zielland
	 * @param x Anzahl Einheiten
	 * @throws NichtGenugEinheitenException , wenn in x > Einheiten aus land1
	 */
	public void verschiebeEinheiten(Land land1, Land land2, int x) throws NichtGenugEinheitenException {
		if(land1.getAnzEinheiten()<=x) throw new NichtGenugEinheitenException(land1);
		this.removeEinheiten(land1, x);
		this.addEinheiten(land2, x);
		land1.setBeteiligt(true);
		land2.setBeteiligt(true);
	}
	
	/**
	 * -> fgt x Einheiten zum Land hinzu
	 * 
	 * @param l Land
	 * @param x Anzahl Einheiten
	 */
	public void addEinheiten (Land l, int x) {
		l.setAnzEinheiten(l.getAnzEinheiten()+x);
	}
	
	/**
	 * -> entefernt x Einheiten aus dem Land
	 * 
	 * @param l Land
	 * @param x Anzahl Einheiten
	 */
	public void removeEinheiten (Land l, int x) {
		l.setAnzEinheiten(l.getAnzEinheiten()-x);
	}
	
	/**
	 * wenn alle lnder eines kontinents den selben besitzer haben -> kontinent erobert
	 * 
	 * @param k Kontinent
	 * @return true, wenn erobert; false, sonst
	 */
	public boolean kontinentErobert(Kontinent k){
		Spieler besitzer = k.getLaenderListe().elementAt(0).getBesitzer();
		for(int i=0; i<k.getLaenderListe().size(); i++) {
			if(!(besitzer.equals(k.getLaenderListe().elementAt(i).getBesitzer()))) {
				return false;
			}
		}
		return true;
	}
	
	/**
	 * Prft alle Kontinente, ob sie erobert wurden.
	 */
	public void pruefKontinente() {
		for(Kontinent k : kontinentenListe) {
			k.setErobert(kontinentErobert(k));
		}
	}
	
	/**
	 * Setzt alle Lnder auf unbeteiligt.
	 */
	public void setUnbeteiligt() {
		for(Land l : laenderListe) {
			l.setBeteiligt(false);
		}
	}
	
	/**
	 * -> setzt spieler als neuen Besitzer des Landes l
	 * 
	 * @param l Land 
	 * @param spieler Spieler
	 */
	public void setBesitzer(Land l, Spieler spieler) {
		l.setBesitzer(spieler);
	}
	
	/**
	 * @return Vektor mit allen Lndern
	 */
	public Vector<Land> getLaenderliste() {
		return this.laenderListe;
	}
	
	/**
	 * Prft ob zwei Lnder benachbart sind
	 * 
	 * @param a ein Land
	 * @param b ein anderes Land
	 * @return true, wenn benachbart
	 */
	public boolean isNachbar(Land a, Land b) {
		int aIdx = laenderListe.indexOf(a);
		int bIdx = laenderListe.indexOf(b);
		if(beziehung[aIdx][bIdx]) {
			return true;
		} else {
			return false;
		}
	}
	
	/**
	 * Gibt Lndervektor mit Nachbarlndern zurck
	 * 
	 * @param a Land dessen Nachbarlnder gesucht werden
	 * @return Vektor mit Nachbarlndern
	 */
	public Vector<Land> getNachbarn(Land a) {
		Vector<Land> erg = new Vector<Land>();
		int aIdx = laenderListe.indexOf(a);
		for(int i=0; i<beziehung[aIdx].length; i++) {
			if(beziehung[aIdx][i]&&i!=aIdx) {
				erg.add(laenderListe.elementAt(i));
			}
		}
		return erg;
	}
	
	/**
	 * Gibt Kontinentenliste zurck.
	 * 
	 * @return Kontinentenliste
	 */
	public Vector<Kontinent> getKontinentenliste() {
		return this.kontinentenListe;
	}
	
}
