package risiko.local.domain;

import java.io.FileNotFoundException;
import java.io.IOException;
import java.util.Collections;
import java.util.Iterator;
import java.util.Random;
import java.util.Vector;

import risiko.local.domain.exceptions.KeineNachbarnException;
import risiko.local.domain.exceptions.LandBeteiligtException;
import risiko.local.domain.exceptions.LandNichtImBesitzException;
import risiko.local.domain.exceptions.NichtBenachbartException;
import risiko.local.domain.exceptions.NichtGenugEinheitenException;
import risiko.local.domain.exceptions.NichtGenugWuerfelException;
import risiko.local.domain.exceptions.VerteidigerUebersteigtAngreiferException;
import risiko.local.persistence.PersMgr;
import risiko.local.valueobjects.BefreiungsMission;
import risiko.local.valueobjects.Kontinent;
import risiko.local.valueobjects.LaenderMission;
import risiko.local.valueobjects.Land;
import risiko.local.valueobjects.Mission;
import risiko.local.valueobjects.Spieler;

/**
 * Haptverwaltungsklasse des Spiels.
 * 
 * @author dgrosche
 * @version 2009-05-25
 *
 */
public class SpielMgr {
	
	private WeltMgr welt;
	private SpielerMgr spieler;
	private MissionsMgr missionen;
	
	private KampfMgr aktiverKampf;
	private KartenMgr karten;
	
	private PersMgr persistenz;
	
	private int phase;

	/**
	 * Konstruktor erstellt Unterverwaltungsklassen:
	 * <br>    - Weltverwaltung
	 * <br>    - Spielerverwaltung
	 * <br>    - Kartenverwaltung
	 * <br>    - Datenspeicherung
	 * 
	 */
	public SpielMgr() {
		persistenz = new PersMgr();
		try {
			welt = new WeltMgr(persistenz);
		} catch(Exception e) {
			System.err.println(e);
			welt = new WeltMgr();
		}
		spieler = new SpielerMgr();
		karten = new KartenMgr(welt.getLaenderliste().size());
	}
	
	/*
	 * Spielstart 
	 */
	
	/**
	 * Fgt neuen Spieler hinzu.
	 * 
	 * @param name Spielername
	 */
	public void addSpieler(String name) {
		spieler.addSpieler(name);
	}
	
	/**
	 * Die Lnder werden gleichmig unter die Spieler aufgeteilt.
	 */
	public void verteileLaender() {
		//Alle Lnder werden in eine Liste nicht verteilter Lnder kopiert
		Vector<Land> freieLaender = new Vector<Land>(welt.getLaenderliste());
		while(freieLaender.size()!=0) {
			for(int i=0; i<spieler.getAllSpieler().size(); i++) {
				if(freieLaender.size()!=0) {
					//Es wird eine Zufallszahl zwischen 0 und der Anzahl nicht verteilter Lnder ermittelt
					Random random = new Random();
					int zahl = random.nextInt(freieLaender.size());
					//Dem Land wird der Besitzer zuegordnet
					freieLaender.elementAt(zahl).setBesitzer(spieler.getAllSpieler().elementAt(i));
					//Die Lnderanzahl des Spielers wird aktualisiert
					int x = spieler.getAllSpieler().elementAt(i).getAnzLaender();
					spieler.getAllSpieler().elementAt(i).setAnzLaender(x+1);
					//Das Land wird aus den freien Lndern genommen
					freieLaender.remove(zahl);
				}
			}
		}
	}
	
	/**
	 * Die Missionen werden erstellt und auf die Spieler aufgeteilt.
	 */
	public void verteileMissionen() {
		//Die Missionsverwaltung wird erstellt
		missionen = new MissionsMgr(welt, spieler.getAllSpieler());
		//Alle Missionen werden in eine Liste nicht verteilter Missionen kopiert
		Vector<Mission> freieMissionen = new Vector<Mission>(missionen.getMissionsliste());
		for(Spieler p : spieler.getAllSpieler()) {
			if(freieMissionen.size()!=0) {
				//Es wird eine Zufallszahl zwischen 0 und der Anzahl nicht verteilter Missionen ermittelt
				Random random = new Random();
				int zahl = random.nextInt(freieMissionen.size());
				Mission m = freieMissionen.elementAt(zahl);
				//Die Mission wird aus den freien Missionen genommen
				freieMissionen.removeElementAt(zahl);
				//Bei einer Befreiungsmission muss geguckt werden, ob sich der Spieler selbst verfolgen soll
				if(m instanceof BefreiungsMission) {
					if(((BefreiungsMission) m).getFeind().equals(p)) {
						//... wenn ja, dann muss eine Alternative gemacht werden
						LaenderMission alternative = new LaenderMission(24,1);
						alternative.setMessage("Befreien Sie 24 Lnder Ihrer Wahl!");
						alternative.setSpieler(p);
						p.setMission(alternative);
						//Die normale Mission wird gestrichen und die Alternative eingefgt
						missionen.getMissionsliste().remove(m);
						missionen.getMissionsliste().add(alternative);
					} else {
						//... wenn nicht, dann ist alles ok.
						m.setSpieler(p);
						p.setMission(m);
					}
				//Bei Lndermissionen, muss die Lnderliste des Spielers hinzugefgt werden
				} else if(m instanceof LaenderMission) {
					m.setSpieler(p);
					p.setMission(m);
					((LaenderMission) m).setLaenderliste(getLaender(p)); 
				} else {
					m.setSpieler(p);
					p.setMission(m);
				}
			}
		}
	}
	
	/**
	 * Missioenen werden ersetzt durch Welterobermissionen
	 */
	public void weltErobern() {
		missionen.getMissionsliste().removeAllElements();
		for(Spieler p : spieler.getAllSpieler()) {
			Mission m = new LaenderMission(welt.getLaenderliste().size(),1);
			m.setMessage("Erobern Sie die Welt!");
			missionen.getMissionsliste().add(m);
			m.setSpieler(p);
			p.setMission(m);
		}
	}
	
	/**
	 * Die Startreihenfolge wird ausgewrfelt.
	 * 
	 * @return Vektor mit Wrfelzahlen
	 */
	public Vector<Integer> reihenfolge() {
		//Das Auswrfeln geschieht ber den KampfMgr
		Vector<Integer> zahlen = new Vector<Integer>();
		KampfMgr wuerfel = new KampfMgr();
		for(int i=0; i<spieler.getAllSpieler().size(); i++) {
			zahlen.add(wuerfel.wuerfel());
		}
		//Die hchste Zahl wird ermittelt
		int hoch = Collections.max(zahlen);
		//Der Spieler mit der hchsten Zahl ist dran
		int idx  = zahlen.indexOf(hoch);
		spieler.getAllSpieler().elementAt(idx).setDran(true);
		//Die Wrfelzahlen werden zur Ausgabe bergeben
		return zahlen;
	}
	
	/*
	 * Spielverlauf
	 */
	
	/**
	 * Setzt entsprechend der Spielphase den Modus weiter.
	 * <br>
	 *   <br><b> Phase 0:   </b> Hauptmen					
	 *   <br><b> Phase 1:   </b> Angriffsmen				
	 *   <br><b> Phase 2:   </b> Verschiebenmen			
	 *   <br><b> Phase 3:   </b> Lnder drucken			
	 *   <br><b> Phase 4:   </b> Zug beenden				
	 *   <br><b> Phase 5:   </b> Karten einlsen			
	 *   <br><b> Phase 6:   </b> Spiel speichern			
	 *   <br><b> Phase 7:   </b> Spiel laden				
	 *   <br><b> Phase 50:  </b> Verteidigen				
	 *   <br><b> Phase 60:  </b> Verstrkung setzen		
	 *   <br><b> Phase 70:  </b> Einheiten nachrcken		
	 *   <br><b> Phase 100: </b> Spiel beenden				
	 * 
	 */
	public void weiter() {
		if(isGewonnen()) this.setPhase(100);
		switch(phase) {
			// 1: Neuer Angriff 
			case 1: this.setPhase(50); break; //weiter bei 50
			// 2: Einheiten verschieben
			case 2: this.setPhase(3); break; //weiter bei 3
			// 3: Zug beenden
			case 3: {
				this.beendeZug();
				//fallse mehr als 4 Karten
				if(getAktiverSpieler().getKarten().size()>4)
					this.setPhase(5);	//weiter bei 5
				else
					this.setPhase(60);	//weiter bei 60
			} break;
			// 4: Lnder anzeigen
			case 4: this.setPhase(0); break; //weiter bei 0 (Hauptmen)
			// 5: Karten einlsen
			case 5: this.setPhase(0); break; //weiter bei 0 (Hauptmen)
			// 6: Spiel speichern
			case 6: this.setPhase(0); break; //weiter bei 0 (Hauptmen)
			// 7: Spiel laden
			case 7: this.setPhase(0); break; //weiter bei 0 (Hauptmen)
			// 50: Verteidigen
			case 50: {
				if(aktiverKampf.getAngreifer().getBesitzer().equals(
				   aktiverKampf.getVerteidiger().getBesitzer()))
						this.setPhase(70);  // weiter bei 70
				else
						this.setPhase(0);	//weiter bei 0 (Hauptmen)
			} break;
			// 60: Verstrkung aufstellen
			case 60: this.setPhase(0); break; //weiter bei 0 (Hauptmen)
			// 70: Zusatzeinheiten nachrcken
			case 70: this.setPhase(0); break;
		}
	}
	
	/**
	 * Setzt die Spielphase auf einen bestimmten Wert
	 * <br>
	 *   <br><b> Phase 0:   </b> Hauptmen					
	 *   <br><b> Phase 1:   </b> Angriffsmen				
	 *   <br><b> Phase 2:   </b> Verschiebenmen			
	 *   <br><b> Phase 3:   </b> Lnder drucken			
	 *   <br><b> Phase 4:   </b> Zug beenden				
	 *   <br><b> Phase 5:   </b> Karten einlsen			
	 *   <br><b> Phase 6:   </b> Spiel speichern			
	 *   <br><b> Phase 7:   </b> Spiel laden				
	 *   <br><b> Phase 50:  </b> Verteidigen				
	 *   <br><b> Phase 60:  </b> Verstrkung setzen		
	 *   <br><b> Phase 70:  </b> Einheiten nachrcken		
	 *   <br><b> Phase 100: </b> Spiel beenden					
	 * 
	 * @param wert Wert auf den Phase gesetzt werden soll
	 */
	public void setPhase(int wert) {
		this.phase = wert;
	}
	
	/**
	 * Gibt die Spielphase zurck
	 * <br>
	 *   <br><b> Phase 0:   </b> Hauptmen					
	 *   <br><b> Phase 1:   </b> Angriffsmen				
	 *   <br><b> Phase 2:   </b> Verschiebenmen			
	 *   <br><b> Phase 3:   </b> Lnder drucken			
	 *   <br><b> Phase 4:   </b> Zug beenden				
	 *   <br><b> Phase 5:   </b> Karten einlsen			
	 *   <br><b> Phase 6:   </b> Spiel speichern			
	 *   <br><b> Phase 7:   </b> Spiel laden				
	 *   <br><b> Phase 50:  </b> Verteidigen				
	 *   <br><b> Phase 60:  </b> Verstrkung setzen		
	 *   <br><b> Phase 70:  </b> Einheiten nachrcken		
	 *   <br><b> Phase 100: </b> Spiel beenden			
	 * 
	 * @return aktuelle Phase
	 */
	public int getPhase() {
		return this.phase;
	}
	
	/**
	 * Startet einen neuen Angriff von a nach b mit x Angreifern <i>(Spielphase: 1)</i>
	 * 
	 * @param a angreifendes Land
	 * @param b verteidigendes Land
	 * @param x Anzahl der Einheiten fr den Angriff
	 * @throws LandNichtImBesitzException , wenn a nicht im Besitz des aktiven Spielers
	 * @throws NichtBenachbartException , wenn a und b nicht benachtbart
	 * @throws NichtGenugEinheitenException , wenn a nur 1 Einheit auf dem Land hat 
	 * @throws NichtGenugWuerfelException , wenn x grer als 3.
	 */
	public void starteAngriff(Land a, Land b, int x) throws LandNichtImBesitzException, NichtBenachbartException, NichtGenugEinheitenException, NichtGenugWuerfelException {
		//Ein neuer Kampf wird erstellt
		if(!(a.getBesitzer().equals(getAktiverSpieler()))) {
			throw new LandNichtImBesitzException(a, this.getAktiverSpieler());
		}
		if(!(welt.isNachbar(a, b))) {
			throw new NichtBenachbartException(a,b);
		}
		if(a.getAnzEinheiten()==1) {
			throw new NichtGenugEinheitenException(a);
		}
		this.aktiverKampf = new KampfMgr();
		if(x>3) {
			throw new NichtGenugWuerfelException(x,3,this.getWuerfelAngreifer());
		}
		//Angreifer, Verteidiger und Angriffseinheiten werden gesetzt
		aktiverKampf.setAngreifer(a);
		aktiverKampf.setVerteidiger(b);
		aktiverKampf.setEinheitenAngriff(x);
	}
	
	/**
	 * Verschiebt x Einheiten von Land a nach Land b <i>(Spielphase: 2)</i>
	 * 
	 * @param a Ursprungsland
	 * @param b Zielland
	 * @param x Anzahl der Einheiten
	 * @throws NichtGenugEinheitenException , wenn nicht genug Einheiten im Land a verbleiben
	 * @throws LandBeteiligtException , wenn eines der Lnder bereits am Spielzug beteiligt war
	 * @throws LandNichtImBesitzException , wenn eines der Lnder nicht im Besitz des aktiven Spielers 
	 * @throws NichtBenachbartException , wenn a und b nicht benachbart sind
	 */
	public void verschiebeEinheiten(Land a, Land b, int x) throws NichtGenugEinheitenException, LandBeteiligtException, LandNichtImBesitzException, NichtBenachbartException {
		//Wenn Spiel in Verschiebe-Phase
		if(this.getPhase()==2) {
			if(!(a.getBesitzer().equals(getAktiverSpieler()))) {
				throw new LandNichtImBesitzException(a, this.getAktiverSpieler());
			}
			if(!(b.getBesitzer().equals(getAktiverSpieler()))) {
				throw new LandNichtImBesitzException(b, this.getAktiverSpieler());
			}
			if(!(welt.isNachbar(a, b))) {
				throw new NichtBenachbartException(a,b);
			}
			//Prfe erst ob eines der Lnder schon beteiligt war
			if(a.isBeteiligt()) throw new LandBeteiligtException(a);
			if(b.isBeteiligt()) throw new LandBeteiligtException(b);
		}
		//Dann verschiebe
		welt.verschiebeEinheiten(a, b, x);
	}

	/**
	 * <br>Setzt nchsten Spieler in der Reihe aktiv.
	 * <br>Fgt Karte hinzu falls im Zug Lnder erobert wurden.
	 * <br><i>(Spielphase: 4)</i>
	 */
	public void beendeZug() {
		karten.addKarte(this.getAktiverSpieler());
		spieler.beendeZug();
		karten.setLandErobert(false);
		welt.setUnbeteiligt();
	}
	
	
	/**
	 * Lst einen Kartensatz ein <i>(Spielphase: 5)</i>
	 * 
	 * @param kartensatz einzulsender Kartensatz
	 */
	public void kartenEinloesen(int[] kartensatz) {
		karten.kartenEinloesen(kartensatz, this.getAktiverSpieler());
	}
	
	/**
	 * Speichert den Spielstand <i>(Spielphase: 6)</i>
	 * 
	 * @return Fehlermeldung oder Besttigung
	 */
	public String speichern() {
		String erg;
		//Fge zu speichernde Objekte hinzu
		Vector<Object> objekte = new Vector<Object>();
		objekte.add(welt);
		objekte.add(spieler);
		try {
			//Speichere ber Dateiverwaltung
			persistenz.speichern(objekte);
			erg = "Speichern erfolgreich.";
		} catch (FileNotFoundException e) {
			erg = "FEHLER: "+e;
		} catch (IOException e) {
			erg = "FEHLER: "+e;
		}
		return erg;
	}
	
	/**
	 * Ldt letzten Spielstand <i>(Spielphase: 7)</i>
	 * 
	 * @return Fehlermeldung oder Besttigung
	 */
	public String laden()  {
		String erg;
		try {
			//Lade Welt- und Spielerverwaltung aus der Persistenz
			persistenz.laden();
			welt = persistenz.getWelt();
			spieler = persistenz.getSpieler();
			//Wenn keine Missionsverwaltung vorhanden...
			if(missionen==null) {
				//... lege eine neue an
				missionen = new MissionsMgr(welt, spieler.getAllSpieler());
				//... setze die Spieler entsprechend
				for(Spieler p : spieler.getAllSpieler()) {
					for(Mission m : missionen.getMissionsliste()) {
						if(m.equals(p.getMission())) {
							m.setSpieler(p);
							p.setMission(m);
						}
					}
				}
			}
			erg = "Laden erfolgreich.";
		} catch (FileNotFoundException e) {
			erg = "FEHLER: "+e;
		} catch (IOException e) {
			erg = "FEHLER: "+e;
		} catch (ClassNotFoundException e) {
			erg = "FEHLER: "+e;
		}

		return erg;
	}
	
	/**
	 * Legt Verteidigung fest und fhrt aktuellen Kampf durch. <i>(Spielphase: 50)</i>
	 * 
	 * @param y Anzahl der Einheiten zur Verteidigung
	 * @throws NichtGenugWuerfelException , wenn mehr als 3 Angreifer oder 2 Verteidiger
	 * @throws VerteidigerUebersteigtAngreiferException , wenn mehr Verteidiger als Angreifer
	 * @throws NichtGenugEinheitenException , wenn fr die Anzahl der Einheiten im Kampf (sowohl Angreifer als auch Verteidiger) nicht gengend Einheiten vorhanden sind.
	 */
	public void verteidige(int y) throws NichtGenugWuerfelException, VerteidigerUebersteigtAngreiferException, NichtGenugEinheitenException {
		aktiverKampf.setEinheitenVerteidigung(y);
		if(aktiverKampf.angriff()) {
			this.karten.setLandErobert(true);
		}
	}
	
	/**
	 * Die zustlichen Einheiten fr aktuellen Spieler werden ermittelt. <i>(Spielphase: 60)</i> 
	 * 
	 * @return Anzahl der zustzlichen Einheiten
	 */
	public int getVerstaerkung() {
		Spieler p = this.getAktiverSpieler();
		//Anzahl der Lnder geteilt durch 3
		int erg = p.getAnzLaender() / 3;	
		//Zusatzpunkte fr eroberten Kontinenten
		for(int i=0; i<welt.getKontinentenliste().size(); i++) {
			Kontinent k = welt.getKontinentenliste().elementAt(i);
			if(welt.kontinentErobert(k)) {
				if(k.getLaenderListe().elementAt(0).getBesitzer().equals(getAktiverSpieler())) {
					erg += k.getPunkt();
				}
			}
		}
		//Zusatzpunkte durch in Vorrunde eingelste Karten
		erg += p.getKartenBonus();
		p.setKartenBonus(0);
		//Wenn alles zusammen, kleiner als 3 wre, setze es auf 3.
		erg = (erg<3) ? 3 : erg;
		return erg;
	}		

	/*
	 * Getter, Setter und weitere Methoden
	 */
	
	/**
	 * Prft ob einlsbare Karten vorhanden sind <i>(Spielphase: 5)</i>
	 * 
	 * @return true, wenn min. ein Satz einlsbarer Karten
	 */
	public boolean pruefKarten() {
		return karten.pruefKarten(this.getAktiverSpieler());
	}
	
	/**
	 * @return the einloesbareKarten
	 */
	public Vector<int[]> getEinloesbareKarten() {
		return karten.getEinloesbareKarten();
	}
	
	/**
	 * Prft ob aktiver Spieler gewonnen hat.
	 * 
	 * @return true, wenn aktiver Spieler seine Mission erfllt hat
	 */
	public boolean isGewonnen() {
		Spieler p = this.getAktiverSpieler();
		//Fr Lndermissionen wird die Lnderliste aktualisiert
		missionen.setLaenderliste(p.getMission(), getLaender(p));
		missionen.pruefMissionen();
		if(p.getMission().isErfuellt()) {
			return true;
		} else {
			return false;
		}
	}
	
	
	/**
	 * Gibt Verteidigerland zurck
	 * 
	 * @return verteidigendes Land
	 */
	public Land getVerteidiger() {
		return aktiverKampf.getVerteidiger();
	}
	
	/**
	 * Gibt Angreiferland zurck
	 * 
	 * @return  angreifendes Land
	 */
	public Land getAngreifer() {
		return aktiverKampf.getAngreifer();
	}
	
	/**
	 * Wrfelzahlen des Angreifers
	 * 
	 * @return Vektor mit Wrfelzahlen
	 */
	public Vector<Integer> getWuerfelAngreifer() {
		return aktiverKampf.getWuerfelAngreifer();
	}
	
	/**
	 * Wrfelzahlen des Verteidigers
	 * 
	 * @return Vektor mit Wrfelzahlen
	 */
	public Vector<Integer> getWuerfelVerteidiger() {
		return aktiverKampf.getWuerfelVerteidiger();
	}
	
	/**
	 * Gibt Anzahl angreifender Einheiten zurck
	 * 
	 * @return angreifende Einheiten
	 */
	public int getEinheitenAngriff() {
		return aktiverKampf.getEinheitenAngriff();
	}
	
	/**
	 * Gibt aktiven Spieler zurck
	 * 
	 * @return aktiven Spieler
	 */
	public Spieler getAktiverSpieler() {
		return spieler.getAktiverSpieler();
	}
	
	/**
	 * Gibt alle Spieler zurck
	 * 
	 * @return Vektor mit allen Spielern
	 */
	public Vector<Spieler> getAllSpieler() {
		return spieler.getAllSpieler();
	}

	
	/**
	 * Gibt komplette Lnderliste zurck
	 * 
	 * @return Vektor mit allen Lndern
	 */
	public Vector<Land> getAllLaender() {
		return this.welt.getLaenderliste();
	}
	
	/**
	 * Gibt alle Lnder eines Spielers zurck
	 * 
	 * @param spieler Spieler
	 * @return Vektor mit Lndern des Spielers
	 */
	public Vector<Land> getLaender(Spieler spieler) {
		Vector<Land> erg = new Vector<Land>();
		for(int i=0; i<welt.getLaenderliste().size(); i++) {
			if(welt.getLaenderliste().elementAt(i).getBesitzer().equals(spieler)) {
				erg.add(welt.getLaenderliste().elementAt(i));
			}
		}
		return erg;
	}

	/**
	 * Gibt alle Nachbarlnder eines Landes zurck
	 * 
	 * @param a Land dessen Nachbarn gesucht sind
	 * @return Vektor mit allen Nachbarn des Landes
	 */
	public Vector<Land> getNachbarn(Land a) {
		return this.welt.getNachbarn(a);
	}
	
	/**
	 * Gibt alle eigenen Nachbarn zurck
	 * 
	 * @param a Land dessen eigene Nachbarn gesucht sind
	 * @return Vektor mit allen benachbarten Lndern mit gleichen Besitzer von Land a
	 * @throws KeineNachbarnException , wenn keine eigenen Nachbarn
	 */
	public Vector<Land> getEigeneNachbarn(Land a) throws KeineNachbarnException {
		Vector<Land> erg = new Vector<Land>();
		Iterator<Land> iter = this.getNachbarn(a).iterator();
		while(iter.hasNext()) {
			Land einLand = iter.next();
			if(einLand.getBesitzer().equals(a.getBesitzer())) {
				erg.add(einLand);
			}
		}
		if(erg.size()==0) throw new KeineNachbarnException(a,false);
		return erg;
	}
	
	/**
	 * Gibt alle fremden Nacharn zurck
	 * 
	 * @param a Land dessen fremde Nachbarn gesucht sind
	 * @throws KeineNachbarnException , wenn keine fremden Nachbarn vorhanden
	 * @return Vektor mit allen benachbarten Lndern mit anderen Besitzer als a
	 */
	public Vector<Land> getFremdeNachbarn(Land a) throws KeineNachbarnException {
		Vector<Land> erg = new Vector<Land>();
		Iterator<Land> iter = this.getNachbarn(a).iterator();
		while(iter.hasNext()) {
			Land einLand = iter.next();
			if(!(einLand.getBesitzer().equals(a.getBesitzer()))) {
				erg.add(einLand);
			}
		}
		if(erg.size()==0) throw new KeineNachbarnException(a,true);
		return erg;
	}
	
	/**
	 * Gibt zurck, ob zwei Lnder benachbart sind
	 * 
	 * @param a erstes Land
	 * @param b zweites Land
	 * @return true, wenn beide Lnder benachbart
	 */
	public boolean isNachbar(Land a, Land b) {
		if(!(a.equals(b))) {
			return welt.isNachbar(a, b);
		} else {
			return false;
		}
	}
	
	/**
	 * Fgt x Einheiten zu Land l hinzu.
	 * 
	 * @param l Land
	 * @param x Anzahl Einheiten
	 */
	public void setzeEinheiten(Land l, int x) {
		welt.addEinheiten(l, x);	
	}

}
