package risiko.local.domain;

import java.util.Collections;
import java.util.Random;
import java.util.Vector;

import risiko.local.domain.exceptions.NichtGenugEinheitenException;
import risiko.local.domain.exceptions.NichtGenugWuerfelException;
import risiko.local.domain.exceptions.VerteidigerUebersteigtAngreiferException;
import risiko.local.valueobjects.Land;

/**
 * KampfMgr ist eine Verwaltungsklasse, die den Wrfelalgorithmus 
 * und die Angriffsfunktion bernimmt.
 * 
 * @author dgrosche
 * @version 2009-05-25
 *
 */
public class KampfMgr {
	
	private Vector<Integer> wuerfelzahlen;
	private Vector<Integer> angreifer;
	private Vector<Integer> verteidiger;
	
	private Land a;
	private Land b;
	
	private int einheitenAngriff;
	private int einheitenVerteidigung;
	
	/**
	 * Konstruktor initialisiert die Wrfelzahlen-Vektoren
	 */
	public KampfMgr() {
		this.wuerfelzahlen = new Vector<Integer>();
		this.angreifer     = new Vector<Integer>();
		this.verteidiger   = new Vector<Integer>();
	}

	/**
	 * Wrfelalgorithmus
	 * 
	 * @return ganze Zahl zwischen 1 und 6.
	 */
	public int wuerfel() {
		Random zufall = new Random();
		int wuerfel = zufall.nextInt();
		wuerfel = Math.abs(wuerfel);
		wuerfel = wuerfel % 6;
		wuerfel += 1;
		this.wuerfelzahlen.add(wuerfel);
		return wuerfel;
	}
	
	/**
	 * Fhrt einen Angriff aus:
	 * <br>  - wrfelt fr jede Einheit
	 * <br>  - vergleicht wer gewinnt und wer verliert
	 * <br>  - zieht entsprechend Einheiten ab
	 * <br>  - prft ob Land erobert wurde
	 * 
	 * @throws NichtGenugEinheitenException , wenn gewhlte Einheiten Zahl der Einheiten im Land bersteigt.
	 * @throws NichtGenugWuerfelException , wenn mehr als 3 Angreifer oder 2 Verteidiger
	 * @throws VerteidigerUebersteigtAngreiferException , wenn mehr Verteidiger als Angreifer
	 * 
	 * @return true, wenn Land erobert wurde
	 */
	public boolean angriff() throws NichtGenugEinheitenException, NichtGenugWuerfelException, VerteidigerUebersteigtAngreiferException {	
	
		//Exceptions zur Einheiten-Anzahl
		if(einheitenAngriff>3) throw new NichtGenugWuerfelException(einheitenAngriff,3,angreifer);
		if(einheitenVerteidigung>2) throw new NichtGenugWuerfelException(einheitenVerteidigung,2,angreifer);
		if(einheitenVerteidigung>einheitenAngriff) throw new VerteidigerUebersteigtAngreiferException(einheitenAngriff,einheitenVerteidigung);
		
		int einheitenA = a.getAnzEinheiten();
		int einheitenB = b.getAnzEinheiten();
		
		// Wrfelzahlen fr Angreifer werden berechnet.
		if(einheitenAngriff>=einheitenA) {
			throw new NichtGenugEinheitenException(a,true,angreifer);
		} else {
			for(int i=0; i<einheitenAngriff; i++) {
				this.angreifer.add(this.wuerfel());
			}
		}
		
		// Wrfelzahlen fr Verteidiger werden berechnet.
		if(einheitenVerteidigung>=einheitenB) {
			if(!(einheitenVerteidigung==einheitenB && einheitenVerteidigung==1)) {
				throw new NichtGenugEinheitenException(b,true,angreifer);
			}
		}
		for(int i=0; i<einheitenVerteidigung; i++) {			
			this.verteidiger.add(this.wuerfel());
		}

		//Zahlen werden numerisch rckwrts sortiert, d.h. grte zuerst
		Collections.sort(angreifer,Collections.reverseOrder());
		Collections.sort(verteidiger,Collections.reverseOrder());
		
		//Die angreifenden Einheiten werden aus dem Land a abgezogen.
		a.setAnzEinheiten(a.getAnzEinheiten()-einheitenAngriff);
		
		for(int i=0; i<this.verteidiger.size(); i++) {

			if(angreifer.elementAt(i)>verteidiger.elementAt(i)) {
				//Angreifer gewinnt
				b.setAnzEinheiten(b.getAnzEinheiten()-1); 
			} else {
				//Verteidiger gewinnt
				einheitenAngriff -= 1;
			}
		}
		
		// Beider Lnder werden als beteiligt markiert
		a.setBeteiligt(true);
		b.setBeteiligt(true);
		
		if(b.getAnzEinheiten()<=0) { //Wenn in B keine Verteidgereinheiten mehr existieren
			//Lnderanzahlen werden aktualisiert
			int x = b.getBesitzer().getAnzLaender();
			b.getBesitzer().setAnzLaender(x-1);
			a.getBesitzer().setAnzLaender(a.getBesitzer().getAnzLaender()+1);
			b.setBesitzer(a.getBesitzer()); //Wechselt der Besitzer
			b.setAnzEinheiten(einheitenAngriff); //Werden die Truppen verschoben
			return true;
		} else { //Sonst kehren die Angreifer nach A zurck.
			a.setAnzEinheiten(a.getAnzEinheiten()+einheitenAngriff);
			return false;
		}
	}
	
	/**
	 * Gibt Angreiferland zurck
	 * 
	 * @return  angreifendes Land
	 */
	public Land getAngreifer() {
		return a;
	}
	
	/**
	 * Setzt Angreiferland
	 * 
	 * @param a angreifendes Land
	 */
	public void setAngreifer(Land a) {
		this.a = a;
	}
	
	/**
	 * Gibt Anzahl angreifender Einheiten zurck
	 * 
	 * @return angreifende Einheiten
	 */
	public int getEinheitenAngriff() {
		return einheitenAngriff;
	}
	
	/**
	 * Setzt Anzahl der angreifenden Einheiten
	 * 
	 * @param x angreifende Einheiten
	 */
	public void setEinheitenAngriff(int x) {
		this.einheitenAngriff = x;
	}
	
	/**
	 * Wrfelzahlen des Angreifers
	 * 
	 * @return Vektor mit Wrfelzahlen
	 */
	public Vector<Integer> getWuerfelAngreifer() {
		return angreifer;
	}
	
	/**
	 * Gibt Verteidigerland zurck
	 * 
	 * @return verteidigendes Land
	 */
	public Land getVerteidiger() {
		return b;
	}
	
	/**
	 * Setzt Verteidigerland
	 * 
	 * @param b verteidigendes Land
	 */
	public void setVerteidiger(Land b) {
		this.b = b;
	}
	
	/**
	 * Gibt Anzahl verteidigender Einheiten zurck
	 * 
	 * @return verteidigende Einheiten
	 */
	public int getEinheitenVerteidigung() {
		return einheitenVerteidigung;
	}
	
	/**
	 * Setzt Anzahl der verteidigenden Einheiten
	 * 
	 * @param y verteidigende Einheiten
	 */
	public void setEinheitenVerteidigung(int y) {
		this.einheitenVerteidigung = y;
	}
	
	/**
	 * Wrfelzahlen des Verteidigers
	 * 
	 * @return Vektor mit Wrfelzahlen
	 */
	public Vector<Integer> getWuerfelVerteidiger() {
		return verteidiger;
	}
	
	/**
	 * Alle Wrfelzahlen des Kampfes
	 * 
	 * @return Vektor mit Wrfelzahlen
	 */
	public Vector<Integer> getWuerfelzahlen() {
		return this.wuerfelzahlen;
	}
			
}
