package game.model;


/**
 * The Card class is used to represent cards in the deck.  The cards have a suit
 * and a rank and are initialized as such
 * 
 * @author jdavidso
 */
public class Card{
    private Integer rank;
    private String suit;
    
    public static final int DIAMONDS = 0;
    public static final int HEARTS = 1;
    public static final int SPADES = 2;
    public static final int CLUBS = 3;
    
    public static final int JACK = 11;
    public static final int QUEEN = 12;
    public static final int KING = 13;
    public static final int ACE = 14;
    
   
    /**
     * A default constructor with rank 0 and unsuited.
     */
    public Card() {
        rank = 0;
        suit = "Unsuited";
    }

    /**
     * This is the constructor most often used to create a card with a given 
     * suit and rank.
     * 
     * @param suit The String representation of a playing card suit
     * @param rank The Integer representation of the card rank.
     */
    public Card(String suit, Integer rank){
        this.rank = rank;
        this.suit = suit;
    }
    
    /**
     * Constructor with an int representation of the suit
     * 
     * @param suit The Integer representation of a playing card suit
     * @param rank The Integer representation of the card rank.
     */
    public Card(Integer suit, Integer rank){
    	this.rank = rank;

    	switch(suit) {
    	case SPADES:
    		this.suit = "Spades";
    		break;
    	case HEARTS:
    		this.suit = "Hearts";
    		break;
    	case DIAMONDS:
    		this.suit = "Diamonds";
    		break;
    	case CLUBS:
    		this.suit = "Clubs";
    		break;            
    	default: this.suit = "Unsuited";
    	}
    }
    
    /**
     * This constructor constructs a card out of just a string.
     * Strings are Rank then Suit, as such As for the Ace of spades, Kh for the
     * king of hearts Qd for the queen of diamonds Jc for the Jack of clubs then
     * Th for the 10 of hearts 9-2 for the other ranks
     * @param cardString A String representation of the card
     */
    public Card(String cardString) {
        switch(cardString.toUpperCase().charAt(0)) {
            case 'A':
                this.rank = 14;
                break;
            case 'K':
                this.rank = 13;
                break;
            case 'Q':
                this.rank = 12;
                break;
            case 'J':
                this.rank = 11;
                break;
            case 'T':
                this.rank = 10;
                break;
            default:
                try {
                    this.rank = new Integer(cardString.substring(0,1));
                } catch (NumberFormatException ex) {
                    this.rank = 0;
                }
        }
        switch(cardString.toLowerCase().charAt(1)) {
            case 's':
                this.suit = "Spades";
                break;
            case 'h':
                this.suit = "Hearts";
                break;
            case 'd':
                this.suit = "Diamonds";
                break;
            case 'c':
                this.suit = "Clubs";
                break;            
            default: this.suit = "Unsuited";
        }
        
    }

    public Card(int both) {
    	
    	if( both > 60 ) {
    		
    		suit = "Clubs";
    		rank = both - 60;
    		
    	} else if (both > 40 ) {
    		
    		suit = "Spades";
    		rank = both - 40;
    		
    	} else if (both > 20 ) {
    		
    		suit = "Hearts";
    		rank = both - 20;
    		
    	} else {
    		
    		suit = "Diamonds";
    		rank = both;
    	}
    }
    
    /**
     * Returns the rank of the Card
     * 
     * @return rank
     */
    public Integer getRank() {
        return rank;
    }

    /**
     * Returns the suit of the Card
     * 
     * @return suit
     */
    public String getSuit() {
        return suit;
    }
    
    /**
     * Returns the suit of the Card as int
     * 
     * @return 0: diamonds</br>
     * 			1: hearts</br>
     * 			2: spades</br>
     * 			3: clubs</br>
     */
    public int getSuitAsInt() {
    	
    	if("DIAMONDS".equalsIgnoreCase(suit)){
//    		DebugOut.showVerboseModel("getSuitAsInt - is DIAMONDS: " + DIAMONDS);
    		return DIAMONDS;
    	}
    	if("HEARTS".equalsIgnoreCase(suit)){
//    		DebugOut.showVerboseModel("getSuitAsInt - is HEARTS: " + HEARTS);
    		return HEARTS;
    	}
    	if("SPADES".equalsIgnoreCase(suit)){
//    		DebugOut.showVerboseModel("getSuitAsInt - is SPADES: " + SPADES);
    		return SPADES;
    	}
    	if("CLUBS".equalsIgnoreCase(suit)){
//    		DebugOut.showVerboseModel("getSuitAsInt - is CLUBS: " + CLUBS);
    		return CLUBS;
    	}

        return -1;
    }

    /**
     * Returns a human readable String representation of the Card given by the 
     * rank then the first character of the suit.  ie: 7d, Ac
     * 
     * @return a String representing the card (Suit and Rank)
     */
    @Override
    public String toString() {
        return printRank() + printSuit();
    }
    
    /**
     * Returns an human easy recognizible int representation of the card.<br>
     * Example:<br>
     * 'suit: +20, val: +val // Ac: 3x20+14=74 // 2d: 0x20+2=2';
     * 
     * @return getSuitAsInt() * 20 + getRank()
     */
    public int toInt() {    	
        return getSuitAsInt() * 20 + getRank();
    }

    /**
     * Returns the first character of the suit in lower case.
     * 
     * @return substring(0,1) suit
     */
    private String printSuit() {
        return suit.substring(0, 1).toLowerCase();
    }

    /**
     * Returns a human readable rank from the rank integer.
     * 
     * @return rank {(0-9, T, J, Q, K, A }
     */
    private String printRank() {
        if (rank < 10) {
            return rank.toString();
        } else {
            switch (rank) {
                case 10:
                    return "T";
                case 11:
                    return "J";
                case 12:
                    return "Q";
                case 13:
                    return "K";
                case 14:
                    return "A";
            }
        }
        return "";
    }
}
