package game;

import java.util.Calendar;

/**
 * Represents an action of a player.</br>
 * blind posting: 'b'</br>
 * folding: 'f'</br>
 * checking: 'c' + amount of change == 0</br>
 * calling: 'c' + amount of change != 0</br>
 * raising: 'r'</br>
 * 
 * @author Witthold/Korol
 */
public class Action {

	/** Time stamp of creation of the action */
	private long timestamp;

	/** 'b' for blind posting; 'f' for folding; 'c' for checking/calling; 'r' for raising; */
	private char action;
	
	/** Amount of money change by this action */
	private int change;
	
	/** Index of player who applied this action */
	private int player;
	
	/** Amount in the pot before the action takes effect */
	private int potBefore;

	/** If the action is an all-in */
	private boolean allIn;


	/**
	 * Generates a new action.
	 * 
	 * @param potBefore
	 * 			Amount in the pot before the action takes effect
	 * @param player
	 * 			Index of player who applied this action
	 * @param action
	 * 			Type of action as char: 'b' for blind posting; 'f' for folding; 'c' for checking/calling; 'r' for raising;
	 * @param change
	 * 			Amount of money change by this action
	 */
	public Action(int potBefore, int player, char action, int change) {

		Calendar now = Calendar.getInstance();
		timestamp = now.getTimeInMillis();

		this.potBefore = potBefore;
		this.player = player;
		this.action = action;
		this.change = change;
	}
	
	/**
	 * Generates a new action.
	 * 
	 * @param action
	 * 			Type of action as char: 'b' for blind posting; 'f' for folding; 'c' for checking/calling; 'r' for raising;
	 * @param change
	 * 			Amount of money change by this action
	 */
	public Action(char action, int change) {

		Calendar now = Calendar.getInstance();
		timestamp = now.getTimeInMillis();
		
		this.potBefore = -1;
		this.player = -1;
		this.action = action;
		this.change = change;
	}


	/**
	 * @return	true if action is a blind post
	 */
	public boolean isBlind() {

		return action == 'b';
	}

	/**
	 * @return	true if action is a check 
	 */
	public boolean isCheck() {

		return (action == 'c' && change == 0);
	}

	/**
	 * @return	true if action is a call 
	 */
	public boolean isCall() {

		return (action == 'c' && change != 0);
	}

	/**
	 * @return	true if action is a raise 
	 */
	public boolean isRaise() {

		return (action == 'r');
	}

	/**
	 * @return	true if action is a fold 
	 */
	public boolean isFold() {

		return (action == 'f');
	}

	/**
	 * @return	The type of action as a char  
	 */

	public char getAction() {
		
		return action; 
	}

	/**
	 * @return	The amount of the change by the action
	 */
	public int getChange() {

		return change;
	}
	
	/**
	 * @return	The time stamp of the creation of the action 
	 */
	public long getTimestamp() {

		return timestamp;
	}

	/**
	 * @return	Amount in the pot before the action takes effect
	 */
	public int getPotBefore() {
		return potBefore;
	}
	
	/**
	 * @return	Index of player who applied this action
	 */
	public int getPlayer() {
		return player;
	}
	
	/**
	 * Marks the action as an all-in action
	 */
	public void setAllIn() {
		allIn = true;
	}

	/**
	 * @return if it is an all-in action
	 */
	public boolean isAllIn() {
		return allIn;
	}

	/**
	 * @return Type of action as an int.<br>
	 * Fold: -1, Check: 0, Call: 1, Raise: 5, Allin: 9
	 */
	public int toInt() {

		int asInt = -11;
		
		if (isFold()) {
			asInt = -1;
		} else if (isCheck()) {
			asInt = 0;
		} else if (isCall()) {
			asInt = 1;
		} else if (isRaise()) {
			asInt = 5;
		} else if (isAllIn()) {
			asInt = 9;
		}
		return asInt;
	}

	/* (non-Javadoc)
	 * @see java.lang.Object#toString()
	 */
	public String toString() {

		return player + ":" + action + change + " ";
	}
}
