package view.j3d;

import java.util.Enumeration;
import java.util.HashMap;
import java.util.Observable;
import java.util.Observer;
import java.util.Vector;

import javax.media.j3d.Alpha;
import javax.media.j3d.Appearance;
import javax.media.j3d.BoundingSphere;
import javax.media.j3d.BranchGroup;
import javax.media.j3d.Group;
import javax.media.j3d.PositionPathInterpolator;
import javax.media.j3d.Transform3D;
import javax.media.j3d.TransformGroup;
import javax.media.j3d.TransparencyAttributes;
import javax.media.j3d.TransparencyInterpolator;
import javax.vecmath.AxisAngle4f;
import javax.vecmath.Color3f;
import javax.vecmath.Point3f;
import javax.vecmath.Quat4f;
import javax.vecmath.Vector3d;
import javax.vecmath.Vector3f;

import model.Gangfeld;
import model.MoveTimer;
import model.Schatz;
import model.SpielMgr;
import model.Spieler;
import exceptions.GangfeldNotFoundException;
import exceptions.NoRandfeldException;

/**
 * Diese Klasse reprsentiert ein Spielbrett mit Randfeld. kopiert und angepasst
 * vom Schachbrett-Bsp. Entspricht dem View in der MVC-Architektur
 * 
 * @author dgrosche, jzimdars, cdiaspero
 * @version final 2010-05-07
 */
public class Spielbrett extends BranchGroup implements Observer {

	// Grafische Eigenschaften
	private float length;
	private float lengthOfOneSquare;
	private float height;

	// Eigenschaften des Szenegraphen
	private Vector<GangfeldBG> squares = new Vector<GangfeldBG>();
	private Vector<Spielfigur> figures = new Vector<Spielfigur>();
	private Vector<SchatzBG> treasures = new Vector<SchatzBG>();
	private Vector3f[][] coords = new Vector3f[9][9];
	private Transform3D[][] transformationen = new Transform3D[9][9];
	private SpielMgr game;

	private Spielfigur spielfigurAnimation;
	private TransformGroup figurTG;

	/**
	 * Erstellen eines Spielbretts mit gegebenen Werten fr die Seitenlnge und
	 * die Hhe
	 * 
	 * @param length
	 *            Lnge des Spielbretts
	 * @param height
	 *            Hhe des Spielbretts
	 * @param game
	 *            Spiellogik
	 */
	protected Spielbrett(float length, float height, SpielMgr game) {
		this.setCapability(BranchGroup.ALLOW_CHILDREN_WRITE);
		this.setCapability(BranchGroup.ALLOW_CHILDREN_EXTEND);

		// Setzen der Werte
		this.game = game;
		this.setLength(length);
		this.setHeight(height);

		lengthOfOneSquare = length / 9 - length / 9 / 2;

		// Erstellen des Spielbrettes
		TransformGroup transGroup;
		for (int x = 4; x >= -4; x--) {
			float X = (x * length / 9 - length / 9 / 2);
			for (int z = -3; z < 6; z++) {
				// Transformationen werden gespeichert
				transformationen[x + 4][z + 3] = new Transform3D();
				float Z = (z * length / 9 - length / 9 / 2);

				// Koordinaten werden gespeichert und Transformationen
				// ausgefhrt
				coords[x + 4][z + 3] = new Vector3f(X, height, Z);
				transformationen[x + 4][z + 3].setTranslation(new Vector3f(X,
						0.0f, Z));

				// TransformGroups sind auswhlbar
				transGroup = new TransformGroup(transformationen[x + 4][z + 3]);
				transGroup.setCapability(TransformGroup.ALLOW_TRANSFORM_READ);
				transGroup.setCapability(TransformGroup.ALLOW_TRANSFORM_WRITE);
				transGroup.setCapability(TransformGroup.ENABLE_PICK_REPORTING);
				transGroup.setCapability(TransformGroup.ALLOW_CHILDREN_WRITE);
				transGroup.setCapability(TransformGroup.ALLOW_CHILDREN_EXTEND);

				// Speichern der GangfelderBG und Aktualisierung des
				// Szenegraphen
				squares.add(new GangfeldBG(this.getGangfeld(x + 4, z + 3),
						length / 9, height));

				// Rotieren des Gangfeldes
				squares.lastElement().rotate(
						this.getGangfeld(x + 4, z + 3).getOrientation());

				transGroup.addChild(squares.lastElement());

				this.addChild(transGroup);
			}
		}

		// Hinzufgen der Schtze zum Spielfeld
		for (Schatz s : game.getSchatzList()) {
			TransformGroup tg = new TransformGroup(transformationen[s
					.getPosition()[0]][s.getPosition()[1]]);
			tg.setCapability(TransformGroup.ALLOW_TRANSFORM_READ);
			tg.setCapability(TransformGroup.ALLOW_TRANSFORM_WRITE);
			tg.setCapability(TransformGroup.ENABLE_PICK_REPORTING);
			tg.setCapability(TransformGroup.ALLOW_CHILDREN_WRITE);
			tg.setCapability(TransformGroup.ALLOW_CHILDREN_EXTEND);
			SchatzBG schatz = new SchatzBG(s, lengthOfOneSquare);
			treasures.add(schatz);
			tg.addChild(schatz);
			this.addChild(tg);
		}

		// Hinzufgen der Spieler zum Spielbrett
		for (Spieler s : game.getAllSpieler()) {
			TransformGroup tg = new TransformGroup(transformationen[s
					.getPosition()[0]][s.getPosition()[1]]);
			tg.setCapability(TransformGroup.ALLOW_TRANSFORM_READ);
			tg.setCapability(TransformGroup.ALLOW_TRANSFORM_WRITE);
			tg.setCapability(TransformGroup.ENABLE_PICK_REPORTING);
			tg.setCapability(TransformGroup.ALLOW_CHILDREN_WRITE);
			tg.setCapability(TransformGroup.ALLOW_CHILDREN_EXTEND);
			Spielfigur f = new Spielfigur(s);
			figures.add(f);
			tg.addChild(f);
			this.addChild(tg);
		}

		this.showWaymarks();

		BorderWall floor = new BorderWall(70.0f, 1.0f, 70.0f, -10.0f, -4.0f,
				10.0f);
		BorderWall wall1 = new BorderWall(1.0f, 4.0f, 70.0f, -80.0f, -1.0f,
				10.0f);
		BorderWall wall2 = new BorderWall(1.0f, 4.0f, 70.0f, 60.0f, -1.0f,
				10.0f);
		BorderWall wall3 = new BorderWall(70.0f, 4.0f, 1.0f, -10.0f, -1.0f,
				80.0f);
		BorderWall wall4 = new BorderWall(70.0f, 4.0f, 1.0f, -10.0f, -1.0f,
				-60.0f);
		this.addChild(floor);
		this.addChild(wall1);
		this.addChild(wall2);
		this.addChild(wall3);
		this.addChild(wall4);

		// Hinzufgen des Views als Observer des Models
		game.addObserver(this);

	}

	/**
	 * Gibt das Gangfeld einer bestimmten Position zurck
	 * 
	 * @param i
	 *            Reihe
	 * @param j
	 *            Spalte
	 * @return Ausgewhltes oder wahlweise erstelltes Gangfeld
	 */
	private Gangfeld getGangfeld(int i, int j) {
		try {
			return game.getGangfeld(i, j);
		} catch (GangfeldNotFoundException e) {
			e.printStackTrace();
			return new Gangfeld(i, j, Gangfeld.FEST);
		}
	}

	/**
	 * Gibt ein Gangfeld anhand seiner BranchGroup zurck
	 * 
	 * @param bg
	 *            BranchGroup Gangfeld
	 * @return Datenobjekt Gangfeld
	 * @throws GangfeldNotFoundException
	 *             , wenn Sucher erfolglos
	 */
	private Gangfeld getGangfeld(GangfeldBG bg)
			throws GangfeldNotFoundException {
		for (Gangfeld sqr : game.getAllGangfelder()) {
			if (bg.equals(sqr))
				return sqr;
			else if (this.getGangfeldBG(sqr).samePos(bg))
				return sqr;
		}
		throw new GangfeldNotFoundException();
	}

	/**
	 * Gibt einen Spieler anhand der Spielfigur zuck
	 * 
	 * @param bg
	 *            BranchGroup der Spielfigur
	 * @return Spieler zur Spielfigur oder null
	 */
	private Spieler getSpieler(Spielfigur bg) {
		for (Spieler sqr : game.getAllSpieler()) {
			if (bg.equals(sqr))
				return sqr;
		}
		return null;
	}

	/**
	 * Gibt einen Schatz anhand der BranchGroup zurck
	 * 
	 * @param bg
	 *            BranchGroup Schatz
	 * @return Datenobjekt Schatz oder null
	 */
	private Schatz getSchatz(SchatzBG bg) {
		for (Schatz s : game.getSchatzList()) {
			if (bg.equals(s))
				return s;
		}
		return null;
	}

	/**
	 * Gibt die BranchGroup eines gewhlten Gangfeldes zurck
	 * 
	 * @param g
	 *            gewhltes Gangfeld
	 * @return zugehriges GangfeldBG
	 */
	private GangfeldBG getGangfeldBG(Gangfeld g) {
		for (GangfeldBG sqr : squares) {
			if (sqr.equals(g))
				return sqr;
		}
		return new GangfeldBG(g, length / 9, height);
	}

	/**
	 * Kollabiert das Spielbrett auf einen Ausgangszustand
	 */
	@SuppressWarnings("unchecked")
	private void collapse() {
		// Lschen berflssiger GangfeldBGs
		Vector<GangfeldBG> toKeep = new Vector<GangfeldBG>();
		for (Gangfeld g : game.getAllGangfelder()) {
			for (GangfeldBG gBG : squares) {
				if (gBG.equals(g))
					toKeep.add(gBG);
			}
		}

		// Aktualisieren des Szenegraphen
		squares.retainAll(toKeep);
		Enumeration children = this.getAllChildren();
		while (children.hasMoreElements()) {
			Object child = children.nextElement();
			if (child instanceof TransformGroup) {
				TransformGroup tg = (TransformGroup) child;
				Transform3D tgTrans = new Transform3D();
				Transform3D t3d = new Transform3D();
				tg.getTransform(tgTrans);
				tgTrans.invert(t3d);
				tg.setTransform(t3d);
				Enumeration transGroup = tg.getAllChildren();
				while (transGroup.hasMoreElements()) {
					Object o = transGroup.nextElement();
					if (o instanceof GangfeldBG) {
						GangfeldBG gBG = (GangfeldBG) o;
						if (!squares.contains(gBG))
							tg.removeChild(gBG);
					}
				}
			}
		}
	}

	/**
	 * Baut das Spielfeld anhand des Model wieder auf
	 */
	private void rebuild() {
		// GangfeldBG des Szenegraphen aktualiseren
		for (GangfeldBG sqr : squares) {
			try {
				TransformGroup tg = this.getTrans(sqr);
				Gangfeld g = this.getGangfeld(sqr);
				tg.setTransform(transformationen[g.getRow()][g.getColumn()]);
			} catch (GangfeldNotFoundException e) {
				e.printStackTrace();
			}
		}

		// Neue Gangfelder zum Szenegraphen hinzufgen
		for (Gangfeld g : game.getAllGangfelder()) {
			GangfeldBG gBG = this.getGangfeldBG(g);
			if (!squares.contains(gBG)) {
				gBG.rotate(g.getOrientation());
				// beachten!
				TransformGroup tg = this.getFreeTrans();
				tg.setTransform(transformationen[g.getRow()][g.getColumn()]);
				tg.addChild(gBG);
				squares.add(gBG);
			}
		}

		// Spieler und Spielfiguren aktualisieren
		for (Spielfigur f : figures) {
			TransformGroup tg = this.getTrans(f);
			Spieler s = this.getSpieler(f);
			if (s.getVisible()) {
				tg.setTransform(transformationen[s.getPosition()[0]][s
						.getPosition()[1]]);
			}
		}

		// Schtze aktualisieren
		for (SchatzBG t : treasures) {
			TransformGroup tg = this.getTrans(t);
			Schatz s = this.getSchatz(t);
			tg.setTransform(transformationen[s.getPosition()[0]][s
					.getPosition()[1]]);
			if (s.getPosition()[0] < 1 || s.getPosition()[0] > 7
					|| s.getPosition()[1] < 1 || s.getPosition()[1] > 7) {
				Transform3D t3d = new Transform3D();
				t3d.setTranslation(new Vector3d(1000.0, 0.0, 0.0));
				tg.setTransform(t3d);
			}
		}
	}

	/**
	 * Markiert ein Randfeld und demarkiert den Rest
	 * 
	 * @param g
	 *            zu markierendes Randfeld
	 */
	@SuppressWarnings("unchecked")
	private void highlight(Group g) {
		Enumeration children = g.getAllChildren();
		while (children.hasMoreElements()) {
			Object child = children.nextElement();
			if (child instanceof GangfeldBG) {
				GangfeldBG gbg = (GangfeldBG) child;
				Gangfeld gf;
				try {
					gf = this.getGangfeld(gbg);
					if (gf.getTyp() == Gangfeld.RAND) {
						g.removeChild(gbg);
						squares.remove(gbg);
						GangfeldBG neu;
						if (gf.isHighlighted())
							neu = new GangfeldBG(gf, length / 9, height, 0.5f);
						else
							neu = new GangfeldBG(gf, length / 9, height, 1f);
						g.addChild(neu);
						squares.add(neu);
					}
				} catch (GangfeldNotFoundException e) {
					e.printStackTrace();
				} catch (NoRandfeldException e) {
					e.printStackTrace();
				}
			} else if (child instanceof Group) {
				this.highlight((Group) child);
			}
		}
	}

	/**
	 * Gibt die bergeordnete TransformGroup eines GangfeldBG zurck
	 * 
	 * @param from
	 *            Kind-GangfeldBG
	 * @return bergeordnete TransformGroup
	 */
	@SuppressWarnings("unchecked")
	private TransformGroup getTrans(GangfeldBG from) {
		Enumeration children = this.getAllChildren();
		while (children.hasMoreElements()) {
			Object child = children.nextElement();
			if (child instanceof TransformGroup) {
				TransformGroup erg = (TransformGroup) child;
				if (erg.getAllChildren().hasMoreElements()) {
					Object vgl = erg.getAllChildren().nextElement();
					if (vgl.equals(from))
						return erg;
				}
			}
		}
		return null;
	}

	/**
	 * Gibt die TransformGroup der Spielfigur zurck
	 * 
	 * @param from
	 *            Spielfigur-BranchGroups
	 * @return bergeordnete TransformGroup
	 */
	@SuppressWarnings("unchecked")
	private TransformGroup getTrans(Spielfigur from) {
		Enumeration children = this.getAllChildren();
		while (children.hasMoreElements()) {
			Object child = children.nextElement();
			if (child instanceof TransformGroup) {
				TransformGroup erg = (TransformGroup) child;
				if (erg.getAllChildren().hasMoreElements()) {
					Object vgl = erg.getAllChildren().nextElement();
					if (vgl.equals(from))
						return erg;
				}
			}
		}
		return null;
	}

	/**
	 * Gibt die TransformGroup eines Schatzes zurck
	 * 
	 * @param from
	 *            Schatz-BranchGroups
	 * @return bergeordnete TransformGroup
	 */
	@SuppressWarnings("unchecked")
	private TransformGroup getTrans(SchatzBG from) {
		Enumeration children = this.getAllChildren();
		while (children.hasMoreElements()) {
			Object child = children.nextElement();
			if (child instanceof TransformGroup) {
				TransformGroup erg = (TransformGroup) child;
				if (erg.getAllChildren().hasMoreElements()) {
					Object vgl = erg.getAllChildren().nextElement();
					if (vgl.equals(from))
						return erg;
				}
			}
		}
		return null;
	}

	/**
	 * @return TransformGroup ohne Kinder
	 */
	@SuppressWarnings("unchecked")
	private TransformGroup getFreeTrans() {
		Enumeration children = this.getAllChildren();
		while (children.hasMoreElements()) {
			Object child = children.nextElement();
			if (child instanceof TransformGroup) {
				TransformGroup erg = (TransformGroup) child;
				if (!erg.getAllChildren().hasMoreElements()) {
					return erg;
				}
			}
		}
		return null;
	}

	/**
	 * Lscht PossibleAreas aus einem Group-Knoten
	 * 
	 * @param g
	 *            Groupknoten
	 */
	@SuppressWarnings("unchecked")
	private void delWaymarks(Group g) {
		Enumeration children = g.getAllChildren();
		while (children.hasMoreElements()) {
			Object child = children.nextElement();
			if (child instanceof PossibleArea) {
				PossibleArea pa = (PossibleArea) child;
				g.removeChild(pa);
			} else if (child instanceof Group) {
				Group c = (Group) child;
				this.delWaymarks(c);
			}
		}
	}

	/**
	 * Fgt GangfeldBGs PossibleAreas hinzu
	 */
	private void showWaymarks() {
		this.delWaymarks(this);
		for (GangfeldBG gbg : squares) {
			try {
				Gangfeld g = this.getGangfeld(gbg);
				if (g.getWaymark())
					gbg.addChild(new PossibleArea(g.getWaymarkColor()));
			} catch (GangfeldNotFoundException e) {
				e.printStackTrace();
			}
		}
	}

	/**
	 * Spielt die Spielfiguranimation ab
	 * 
	 * @param path
	 *            Liste mit Wegpunkten
	 */
	private void playFigureAnimation(int[][] path) {
		BranchGroup objRoot = new BranchGroup();

		int increasingAlphaDuration = 1250 * (path.length - 1);
		new MoveTimer(increasingAlphaDuration, null, this);
		Alpha alpha = new Alpha(1, Alpha.INCREASING_ENABLE, 1000, 0,
				increasingAlphaDuration, 0, 0, 0, 0, 10000);

		// abstimmung der systemzeit mit der animation
		alpha.setStartTime(System.currentTimeMillis() - alpha.getTriggerTime());
	
		TransformGroup target = new TransformGroup();
		target.setCapability(TransformGroup.ALLOW_TRANSFORM_WRITE);
		Transform3D axisOfRotPos = new Transform3D();
		AxisAngle4f axis = new AxisAngle4f(0.0f, 0.0f, 0.0f, 0.0f);
		axisOfRotPos.set(axis);

		float[] knots = new float[path.length];
		float knotDivider = 1.0f / (path.length - 1);
		for (int i = 0; i < path.length; i++) {
			if (i != 0) {
				knots[i] = knotDivider * i;
			} else {
				knots[i] = 0;
			}
		}

		Quat4f[] quats = new Quat4f[path.length];

		for (int i = 0; i < quats.length; i++) {
			quats[i] = new Quat4f(0.0f, 0.0f, 0.0f, 0.0f);
		}

		Point3f[] positions = new Point3f[path.length];

		for (int i = 0; i < positions.length; i++) {
			positions[i] = new Point3f(coords[path[i][0]][path[i][1]].x, 0.0f,
					coords[path[i][0]][path[i][1]].z);
		}

		PositionPathInterpolator posPath = new PositionPathInterpolator(alpha,
				target, axisOfRotPos, knots, positions);

		posPath.setSchedulingBounds(new BoundingSphere());

		objRoot.addChild(target);
		objRoot.addChild(posPath);

		spielfigurAnimation = new Spielfigur(new Color3f(game
				.getCurrentPlayer().getColor3f()));
		spielfigurAnimation.setCapability(BranchGroup.ALLOW_CHILDREN_WRITE);
		spielfigurAnimation.setCapability(BranchGroup.ALLOW_CHILDREN_EXTEND);
		spielfigurAnimation.setCapability(BranchGroup.ALLOW_BOUNDS_WRITE);
		spielfigurAnimation.setCapability(BranchGroup.ALLOW_DETACH);
		figurTG = new TransformGroup();

		figurTG.addChild(spielfigurAnimation);
		target.addChild(figurTG);

		this.addChild(objRoot);
	}

	/**
	 * Spielt die Gangfeld-Animation ab
	 * 
	 * @param gangfelder
	 *            Tabelle mit Gangfelder
	 * @throws GangfeldNotFoundException
	 *             wenn Postionsgangfeld nicht am Rand
	 */
	private void playGangfeldAnimation(HashMap<String, Gangfeld> gangfelder)
			throws GangfeldNotFoundException {
		BranchGroup objRoot = new BranchGroup();
		objRoot.setCapability(BranchGroup.ALLOW_CHILDREN_WRITE);
		objRoot.setCapability(BranchGroup.ALLOW_DETACH);
		GangfeldBG pos = this.getGangfeldBG(gangfelder.get("Position"));
		GangfeldBG field = this.getGangfeldBG(gangfelder.get("Einschub"));
		field.rotate(this.getGangfeld(field).getOrientation());
		for (Appearance app : pos.getAllAppearence(pos)) {
			app.setTransparencyAttributes(new TransparencyAttributes(
					TransparencyAttributes.BLENDED, 1f));
		}
		Alpha a1 = new Alpha(1, Alpha.INCREASING_ENABLE, 10000, 0, 3000, 0,
				1000, 0, 0, 0); // 10,0,5,1,0,0,0

		// Systemzeit mit Animation abstimmen
		a1.setStartTime(System.currentTimeMillis() - a1.getTriggerTime());

		TransparencyAttributes t1 = new TransparencyAttributes(
				TransparencyAttributes.BLENDED, 1.0f);
		t1.setCapability(TransparencyAttributes.ALLOW_VALUE_WRITE);

		TransparencyInterpolator transInt = new TransparencyInterpolator(a1, t1);
		transInt.setSchedulingBounds(new BoundingSphere());
		transInt.setMinimumTransparency(1f);
		transInt.setMaximumTransparency(0f);

		for (Appearance app : field.getAllAppearence(field)) {
			app.setTransparencyAttributes(t1);
		}
		objRoot.addChild(transInt);
		objRoot.addChild(field);
		this.getTrans(pos).addChild(objRoot);

		while (!a1.finished())
			;

		this.getTrans(pos).removeChild(objRoot);

	}

	/**
	 * @param row
	 *            Reihe des gesuchten Feldes
	 * @param column
	 *            Spalte des gesuchten Feldes
	 * @return Koordinaten als Vektor
	 */
	protected Vector3f getCoordOf(int row, int column) {
		if ((row >= 0) && (row <= 9) && (column >= 0) && (column <= 9))
			return coords[row][column];
		else
			return null;
	}

	/**
	 * @param length
	 *            the length to set
	 */
	protected void setLength(float length) {
		this.length = length;
	}

	/**
	 * @return the length
	 */
	protected float getLength() {
		return length;
	}

	/**
	 * @param height
	 *            the height to set
	 */
	protected void setHeight(float height) {
		this.height = height;
	}

	/**
	 * @return the height
	 */
	protected float getHeight() {
		return height;
	}

	/**
	 * Wertet eine Aktualisierung des Model aus
	 * 
	 * @param arg0
	 *            aktualisiertes Model
	 * @param arg1
	 *            zustzliches bergabeobjekt
	 */
	@SuppressWarnings("unchecked")
	@Override
	public void update(Observable arg0, Object arg1) {
		if (arg0.equals(game)) {
			if (arg1 != null && game.isWayCoordsSet()) {
				this.playFigureAnimation((int[][]) arg1);
				game.setWayCoords(false);
				game.notifyObservers();
	
			} else if (arg1 instanceof HashMap) {
				try {
					this.playGangfeldAnimation((HashMap<String, Gangfeld>) arg1);
					game.setWayCoords(false);
					game.notifyObservers();
				} catch (GangfeldNotFoundException e) {
					e.printStackTrace();
				}
			} else {
				this.collapse();
				this.rebuild();
				this.showWaymarks();
				this.highlight(this);
				if (!game.getAnimationSet()) {
					game.getCurrentPlayer().setVisible(true);
					Vector<Spieler> s = game.getAllSpieler();
					for (int i = 0; i < s.size(); i++) {
						s.get(i).setVisible(true);
					}
					for (Spielfigur f : figures) {
						TransformGroup tg = this.getTrans(f);
						Spieler p = this.getSpieler(f);
						tg.setTransform(transformationen[p.getPosition()[0]][p
								.getPosition()[1]]);
					}
					if (spielfigurAnimation != null)
						spielfigurAnimation.setDeleted();
					game.setAnimation(true);
				}
			}
		}
	}

	/**
	 * Lscht die Spielfigur-Animation
	 */
	public void deleteAnimation() {
		for (Spielfigur f : figures) {
			TransformGroup tg = this.getTrans(f);
			Spieler p = this.getSpieler(f);
			tg.setTransform(transformationen[p.getPosition()[0]][p
					.getPosition()[1]]);
		}
		if (spielfigurAnimation != null)
			spielfigurAnimation.setDeleted();
	}

}