package view.j3d;

import java.io.FileNotFoundException;
import java.io.FileReader;
import java.util.Map;

import javax.media.j3d.BranchGroup;
import javax.media.j3d.Shape3D;
import javax.media.j3d.Transform3D;
import javax.media.j3d.TransformGroup;
import javax.vecmath.Vector3d;

import model.Schatz;
import view.j3d.treasureGraphics.Book;
import view.j3d.treasureGraphics.Crown;
import view.j3d.treasureGraphics.Diamond;
import view.j3d.treasureGraphics.Key;
import view.j3d.treasureGraphics.MagicWand;
import view.j3d.treasureGraphics.Ring;
import view.j3d.treasureGraphics.Schatzkiste;

import com.sun.j3d.loaders.IncorrectFormatException;
import com.sun.j3d.loaders.ParsingErrorException;
import com.sun.j3d.loaders.Scene;
import com.sun.j3d.loaders.objectfile.ObjectFile;

/**
 * BranchGriup zur Darstellung von Schtzen
 * 
 * @author jzimdars, dgrosche
 * @version final 2010-05-07
 * 
 */
public class SchatzBG extends BranchGroup {

	// Eigenschaften der Figur
	private double hoehe = 2.0;
	private double breite = 3.0;

	// Berechnete Eigenschaften
	private Schatz schatz;

	// Konstruktoren
	/**
	 * Erstellt eine neue BranchGroup aus einem Schatz
	 * 
	 * @param schatz
	 *            zu erstellender Schatz
	 */
	protected SchatzBG(Schatz schatz) {
		this.schatz = schatz;
		this.setGroesse(this.hoehe, this.breite);
		this.createFigur();
	}

	/**
	 * Erstellt einen neuen Schatz einer spezifische Gre
	 * 
	 * @param schatz
	 *            zu erstellender Schatz
	 * @param dimension
	 *            zu verwendende Gre
	 */
	protected SchatzBG(Schatz schatz, double dimension) {
		this.schatz = schatz;
		this.setGroesse(0.6 * dimension, 0.9 * dimension);
		this.createFigur();
	}

	/**
	 * Fgt der BG die Grundfigur aus Standardmaterial hinzu
	 */
	private void createFigur() {
		String name = schatz.getName();
		if (name.equals("Schatzkiste")) {
			this.createModelFigure("Schatzkiste");
		} else if (name.equals("Diamant")) {
			this.createModelFigure("Diamond");
		} else if (name.equals("Krone")) {
			this.createModelFigure("Crown", 4f);
		} else if (name.equals("Zauberstab")) {
			this.createModelFigure("MagicWand", 10f);
		} else if (name.equals("Ring")) {
			this.createModelFigure("Ring", 7.5f);
		} else if (name.equals("Schlssel")) {
			this.createModelFigure("Key", 4.5f);
		} else if (name.equals("Buch")) {
			this.createModelFigure("Book", 3.5f);
		}
	}

	/**
	 * Fgt der BG einen geladenen Schatz hinzu
	 * 
	 * @param model
	 *            zu verwendendes Objektmodell
	 */
	private void createModelFigure(String model) {
		this.createModelFigure(model, 5f);
	}

	/**
	 * Erstellt einen Schatz aus einem geladenen Model
	 * 
	 * @param model
	 *            zu verwendendes Model
	 * @param scale
	 *            Grenskalierung
	 */
	private void createModelFigure(String model, float scale) {

		Transform3D t3d = new Transform3D();
		t3d.setTranslation(new Vector3d(0.0, 1.5, 0.0));
		TransformGroup tg1 = new TransformGroup(t3d);

		Transform3D skalieren = new Transform3D();
		skalieren.setScale(scale);
		TransformGroup tg2 = new TransformGroup(skalieren);
		tg2.addChild(tg1);
		if (model.equals("Ring")) {
			tg1.addChild(new Ring());
		} else if (model.equals("Schatzkiste")) {
			tg1.addChild(new Schatzkiste());
		} else if (model.equals("Book")) {
			tg1.addChild(new Book());
		} else if (model.equals("Key")) {
			tg1.addChild(new Key());
		} else if (model.equals("Diamond")) {
			tg1.addChild(new Diamond());
		} else if (model.equals("MagicWand")) {
			tg1.addChild(new MagicWand());
		} else if (model.equals("Crown")) {
			tg1.addChild(new Crown());
		} else {
			BranchGroup objBg = loadObject("objects/" + model + ".obj");
			tg1.addChild(objBg);
		}

		this.addChild(tg2);
	}

	/**
	 * Setzt die Gre der Figur
	 * 
	 * @param hoehe
	 *            Hhe
	 * @param breite
	 *            Breite
	 */
	private void setGroesse(double hoehe, double breite) {
		this.hoehe = hoehe;
		this.breite = breite;
	}

	/**
	 * Ldt ein Objekt aus einer Datei
	 * 
	 * @param filename
	 *            Dateiname
	 * @return geladenen Knoten
	 */
	@SuppressWarnings("unchecked")
	private BranchGroup loadObject(String filename) {
		ObjectFile loader = new ObjectFile(ObjectFile.RESIZE);

		Scene scene = null;

		try {
			scene = loader.load(new FileReader(filename));
		} catch (FileNotFoundException e) {
			e.printStackTrace();
		} catch (IncorrectFormatException e) {
			e.printStackTrace();
		} catch (ParsingErrorException e) {
			e.printStackTrace();
		}
		Map<String, Shape3D> nameMap = scene.getNamedObjects();
		for (String name : nameMap.keySet()) {
			nameMap.get(name);
		}
		return scene.getSceneGroup();
	}

	/**
	 * Vergleicht SchattzBG mit einem Schatz
	 */
	@Override
	public boolean equals(Object o) {
		if (o instanceof Schatz) {
			Schatz s = (Schatz) o;
			if (s.equals(schatz))
				return true;
			else
				return false;
		} else
			return super.equals(o);
	}

}
