package view.j3d;

import java.util.Enumeration;
import java.util.Vector;

import javax.media.j3d.Appearance;
import javax.media.j3d.BranchGroup;
import javax.media.j3d.Group;
import javax.media.j3d.Material;
import javax.media.j3d.Shape3D;
import javax.media.j3d.Texture;
import javax.media.j3d.TextureAttributes;
import javax.media.j3d.Transform3D;
import javax.media.j3d.TransformGroup;
import javax.media.j3d.TransparencyAttributes;
import javax.vecmath.Color3f;
import javax.vecmath.Color4f;
import javax.vecmath.Vector3d;

import model.Gangfeld;

import com.sun.j3d.utils.geometry.Box;
import com.sun.j3d.utils.geometry.Primitive;

import exceptions.NoRandfeldException;

/**
 * BranchGroup zur Darstellung von Gangfeldern
 * 
 * @author dgrosche, jzimdars
 * @version final 2010-05-07
 * 
 */
public class GangfeldBG extends BranchGroup {

	// interne Eigenschaften
	private float length, height, thickness;
	private Box grundflaeche;
	private Gangfeld gangfeld;
	
	private Material mat;
	private Appearance app;
	private Texture texture;
	
	private boolean visibleGangfeld = false;

	/**
	 * Erstellt eine neue BranchGroup mit dem gewnschten Gangfeld
	 * 
	 * @param gangfeld
	 *            Gangfeld, das erzeugt werden soll
	 * @param laenge
	 *            Wie lang soll das Gangfeld sein (bernommen aus Bsp.)
	 * @param hoehe
	 *            Wie hoch soll das Spielbrett (nicht die Mauer) sein
	 */
	protected GangfeldBG(Gangfeld gangfeld, float laenge, float hoehe) {
		// bergeben der Werte
		this.length = laenge;
		this.height = hoehe;
		this.thickness = this.length / 5;
		this.gangfeld = gangfeld;

		// Capabilities setzen
		this.setCapability(BranchGroup.ALLOW_DETACH);
		this.setCapability(TransformGroup.ALLOW_TRANSFORM_READ);
		this.setCapability(TransformGroup.ALLOW_TRANSFORM_WRITE);
		this.setCapability(TransformGroup.ENABLE_PICK_REPORTING);
		this.setCapability(TransformGroup.ALLOW_CHILDREN_WRITE);
		this.setCapability(TransformGroup.ALLOW_CHILDREN_EXTEND);

		// Weg-Textur fr Grundflche setzen
		texture = getTexture("images/steine1.jpg");
		TextureAttributes texAttr = getTextureAttributes();
		app = new Appearance();
		app.setTexture(texture);
		app.setTextureAttributes(texAttr);
		app.setCapability(Appearance.ALLOW_MATERIAL_WRITE);
		app.setCapability(Appearance.ALLOW_TEXTURE_ATTRIBUTES_WRITE);
		app.setCapability(Appearance.ALLOW_TRANSPARENCY_ATTRIBUTES_READ);
		app.setCapability(Appearance.ALLOW_TRANSPARENCY_ATTRIBUTES_WRITE);
		int primflags = Primitive.GENERATE_NORMALS
				+ Primitive.GENERATE_TEXTURE_COORDS;

		// Material erstellen
		mat = new Material(
				new Color3f(0.0f, 0.0f, 0.1f), 
				new Color3f(0.01f, 0.01f, 0.01f), 
				new Color3f(0.01f, 0.01f, 0.01f),
				new Color3f(0.3f, 0.3f, 0.3f), 10.0f);
		
		app.setMaterial(mat);

		grundflaeche = new Box(this.length / 2, this.height / 2,
				this.length / 2, primflags, app);
		this.addChild(grundflaeche);
		grundflaeche.setCapability(Box.ENABLE_APPEARANCE_MODIFY);
		grundflaeche.setCapability(Shape3D.ALLOW_APPEARANCE_WRITE);
		grundflaeche.setCapability(Shape3D.ALLOW_APPEARANCE_OVERRIDE_WRITE);
		
		// Je nach Typ sollen die Mauern gesetzt werden
		switch (gangfeld.getTyp()) {
		case (Gangfeld.KURVE):
			this.createKurve();
			break;
		case (Gangfeld.GERADE):
			this.createGerade();
			break;
		case (Gangfeld.TSTUECK):
			this.createTStueck();
			break;
		case (Gangfeld.RAND): {
			this.createTransparent(0.9f);
		}
		}
		visibleGangfeld = true;
	}

	protected GangfeldBG(Gangfeld g, float laenge, float hoehe,
			float transparency) throws NoRandfeldException {
		if (g.getTyp() == Gangfeld.RAND) {
			// bergeben der Werte
			this.length = laenge;
			this.height = hoehe;
			this.thickness = this.length / 5;
			this.gangfeld = g;
			this.setCapability(BranchGroup.ALLOW_DETACH);
			this.setCapability(TransformGroup.ALLOW_TRANSFORM_READ);
			this.setCapability(TransformGroup.ALLOW_TRANSFORM_WRITE);
			this.setCapability(TransformGroup.ENABLE_PICK_REPORTING);
			this.setCapability(TransformGroup.ALLOW_CHILDREN_WRITE);
			this.setCapability(TransformGroup.ALLOW_CHILDREN_EXTEND);

			mat = new Material(
					new Color3f(0.0f, 0.0f, 0.1f), 
					new Color3f(0.01f, 0.01f, 0.01f), 
					new Color3f(0.01f, 0.01f, 0.01f),
					new Color3f(0.3f, 0.3f, 0.3f), 10.0f);
			
			app = new Appearance();
			app.setCapability(Appearance.ALLOW_MATERIAL_WRITE);
			app.setCapability(Appearance.ALLOW_TRANSPARENCY_ATTRIBUTES_READ);
			app.setCapability(Appearance.ALLOW_TRANSPARENCY_ATTRIBUTES_WRITE);
			app.setTransparencyAttributes(new TransparencyAttributes(
					TransparencyAttributes.BLENDED, transparency));

			grundflaeche = new Box(this.length / 2, this.height / 2,
					this.length / 2, app);
			this.addChild(grundflaeche);
			grundflaeche.setCapability(Box.ENABLE_APPEARANCE_MODIFY);
			grundflaeche.setCapability(Shape3D.ALLOW_APPEARANCE_WRITE);
			grundflaeche.setCapability(Shape3D.ALLOW_APPEARANCE_OVERRIDE_WRITE);
			
		} else {
			throw new NoRandfeldException();
		}
	}

	/**
	 * @param filename
	 *            Dateipfad des Texturbildes
	 * @return Texturobjekt
	 */
	private Texture getTexture(String filename) {
		LabTextureLoader ltl = new LabTextureLoader(filename);
		Texture texture = ltl.getImg().getTexture();
		texture.setBoundaryModeS(Texture.WRAP);
		texture.setBoundaryModeT(Texture.WRAP);
		texture.setBoundaryColor(new Color4f(0.0f, 10.0f, 0.0f, 0.0f));
		return texture;
	}

	/**
	 * @return Texturattribute
	 */
	private TextureAttributes getTextureAttributes() {
		TextureAttributes texAttr = new TextureAttributes();
		texAttr.setTextureMode(TextureAttributes.REPLACE);
		return texAttr;
	}

	/**
	 * Erstellt ein transparentes Randfeld
	 */
	private void createTransparent(float transperency) {
		Appearance app = new Appearance();
		app.setCapability(Appearance.ALLOW_TRANSPARENCY_ATTRIBUTES_READ);
		app.setCapability(Appearance.ALLOW_TRANSPARENCY_ATTRIBUTES_WRITE);
		app.setTransparencyAttributes(new TransparencyAttributes(
				TransparencyAttributes.BLENDED, transperency));
		grundflaeche.setAppearance(app);
	}

	/**
	 * Erstellt ein neues T-Stck, dass links geschlossen ist
	 */
	private void createTStueck() {

		Texture texture = getTexture("images/brick.jpg");
		TextureAttributes texAttr = getTextureAttributes();

		Appearance app = new Appearance();
		app.setCapability(Appearance.ALLOW_TRANSPARENCY_ATTRIBUTES_READ);
		app.setCapability(Appearance.ALLOW_TRANSPARENCY_ATTRIBUTES_WRITE);
		app.setTexture(texture);
		app.setTextureAttributes(texAttr);
		int primflags = Primitive.GENERATE_NORMALS
				+ Primitive.GENERATE_TEXTURE_COORDS;

		app.setMaterial(new Material(new Color3f(0.35f, 0.24f, 0.0f),
				new Color3f(0.0f, 0.0f, 0.0f), new Color3f(0.73f, 0.25f, 0.0f),
				new Color3f(0.0f, 0.0f, 0.0f), 0.2f));
		// Die Box erstellt sich im Ursprung mit den Dimensionswerten gespiegelt
		// an den Achsen
		Box wand1 = new Box(thickness / 2, length / 3, length / 2, primflags,
				app); // lange Wand
		Box wand2 = new Box(thickness / 2, length / 3, thickness / 2,
				primflags, app); // obere kleine Wand
		Box wand3 = new Box(thickness / 2, length / 3, thickness / 2,
				primflags, app); // untere kleine Wand

		// Die lange Wand wird an den linken Rand geschoben
		Transform3D t3d1 = new Transform3D();
		t3d1.setTranslation(new Vector3d(thickness / 2 - length / 2, length
				/ 1000 + length / 3, 0));
		TransformGroup tg1 = new TransformGroup(t3d1);
		tg1.addChild(wand1);
		this.addChild(tg1);

		/*
		 * Anmerkung zum Koordinatensystem: x-Achse zeigt nach rechts y-Achse
		 * zeigt nach oben z-Achse zeigt nach hinten
		 */

		// Die obere kleine Wand wird in die obere Ecke geschoben
		Transform3D t3d2 = new Transform3D();
		// Die Verschiebung nach y ist etwas hher als die Hlfte, damit man auf
		// der Unterseite nicht die Wand sieht.
		t3d2.setTranslation(new Vector3d(length / 2 - thickness / 2, length
				/ 1000 + length / 3, thickness / 2 - length / 2));
		TransformGroup tg2 = new TransformGroup(t3d2);
		tg2.addChild(wand2);
		this.addChild(tg2);

		// Die untere kleine Wand wird in die untere Ecke geschoben
		Transform3D t3d3 = new Transform3D();
		t3d3.setTranslation(new Vector3d(length / 2 - thickness / 2, length
				/ 1000 + length / 3, length / 2 - thickness / 2));
		TransformGroup tg3 = new TransformGroup(t3d3);
		tg3.addChild(wand3);
		this.addChild(tg3);
	}

	/**
	 * Erstellt ein neues Geradenstck von unten nach oben
	 */
	private void createGerade() {

		Texture texture = getTexture("images/brick.jpg");
		TextureAttributes texAttr = getTextureAttributes();

		Appearance app = new Appearance();
		app.setCapability(Appearance.ALLOW_TRANSPARENCY_ATTRIBUTES_READ);
		app.setCapability(Appearance.ALLOW_TRANSPARENCY_ATTRIBUTES_WRITE);
		app.setTexture(texture);
		app.setTextureAttributes(texAttr);
		int primflags = Primitive.GENERATE_NORMALS
				+ Primitive.GENERATE_TEXTURE_COORDS;

		app.setMaterial(new Material(new Color3f(0.35f, 0.24f, 0.0f),
				new Color3f(0.0f, 0.0f, 0.0f), new Color3f(0.73f, 0.25f, 0.0f),
				new Color3f(0.0f, 0.0f, 0.0f), 0.2f));
		Box wand1 = new Box(thickness / 2, length / 3, length / 2, primflags,
				app); // linke Wand
		Box wand2 = new Box(thickness / 2, length / 3, length / 2, primflags,
				app); // rechte Wand

		// Linke Wand an linken Rand
		Transform3D t3d1 = new Transform3D();
		t3d1.setTranslation(new Vector3d(thickness / 2 - length / 2, length
				/ 1000 + length / 3, 0));
		TransformGroup tg1 = new TransformGroup(t3d1);
		tg1.addChild(wand1);
		this.addChild(tg1);

		// Rechte Wand an rechten Rand
		Transform3D t3d2 = new Transform3D();
		t3d2.setTranslation(new Vector3d(length / 2 - thickness / 2, length
				/ 1000 + length / 3, 0));
		TransformGroup tg2 = new TransformGroup(t3d2);
		tg2.addChild(wand2);
		this.addChild(tg2);
	}

	/**
	 * Erstellt eine neue Kurve von links nach oben
	 */
	private void createKurve() {

		Texture texture = getTexture("images/brick.jpg");
		TextureAttributes texAttr = getTextureAttributes();

		Appearance app = new Appearance();
		app.setCapability(Appearance.ALLOW_TRANSPARENCY_ATTRIBUTES_READ);
		app.setCapability(Appearance.ALLOW_TRANSPARENCY_ATTRIBUTES_WRITE);
		app.setTexture(texture);
		app.setTextureAttributes(texAttr);
		int primflags = Primitive.GENERATE_NORMALS
				+ Primitive.GENERATE_TEXTURE_COORDS;

		app.setMaterial(new Material(new Color3f(0.35f, 0.24f, 0.0f),
				new Color3f(0.0f, 0.0f, 0.0f), new Color3f(0.73f, 0.25f, 0.0f),
				new Color3f(0.0f, 0.0f, 0.0f), 0.2f));
		Box wand1 = new Box(thickness / 2, length / 3, length / 2, primflags,
				app); // Rechte Wand
		Box wand2 = new Box(thickness / 2, length / 3, length / 2, primflags,
				app); // Untere Wand
		Box wand3 = new Box(thickness / 2, length / 3, thickness / 2,
				primflags, app); // Kleine Wand
		// Rechte Wand an rechten Rand
		Transform3D t3d1 = new Transform3D();
		t3d1.setTranslation(new Vector3d(length / 2 - thickness / 2, length
				/ 1000 + length / 3, 0));
		TransformGroup tg1 = new TransformGroup(t3d1);
		tg1.addChild(wand1);
		this.addChild(tg1);

		// Untere Wand drehen
		Transform3D t3d2 = new Transform3D();
		t3d2.rotY(Math.toRadians(90));
		TransformGroup tg2 = new TransformGroup(t3d2);
		tg2.addChild(wand2);

		// Untere Wand an unteren Rand
		Transform3D t3d3 = new Transform3D();
		t3d3.setTranslation(new Vector3d(0, length / 1000 + length / 3, length
				/ 2 - thickness / 2));
		TransformGroup tg3 = new TransformGroup(t3d3);
		tg3.addChild(tg2);
		this.addChild(tg3);

		// Kleine Wand nach oben links
		Transform3D t3d4 = new Transform3D();
		t3d4.setTranslation(new Vector3d(thickness / 2 - length / 2, length
				/ 1000 + length / 3, thickness / 2 - length / 2));
		TransformGroup tg4 = new TransformGroup(t3d4);
		tg4.addChild(wand3);
		this.addChild(tg4);
	}

	/**
	 * @return Lnge des Feldes
	 */
	protected float getLength() {
		return length;
	}

	/**
	 * @return Hhe des Feldes (nicht der Mauer)
	 */
	protected float getHeight() {
		return height;
	}

	/**
	 * Rotiert ein Feld um ein vielfaches von 90 Grad
	 * 
	 * @param drehung
	 *            Ganzzahl zwischen 0 und 4 oder Gangfeld.LINKS fr Linksdrehung
	 *            Gangfeld.RECHTS fr Rechtsdrehung Gangfeld.WENDE fr
	 *            180-Grad-Drehung
	 */
	protected void rotate(int drehung) {
		// erstellt Translation um Vielfache von 90 Grad
		Transform3D rotation = new Transform3D();
		rotation.rotY(drehung * Math.toRadians(90));
		TransformGroup tg = new TransformGroup(rotation);
		// Das bisherige Feld wird erst geklont und dann gedreht
		tg.addChild(this.cloneTree());
		// Dann wird das bisherige Feld durch das gedrehte ersetzt
		this.removeAllChildren(); // Ntig, da sonst altes noch sichtbar
		this.addChild(tg);
		this.compile(); // d.h. nderungen werden angewandt
	}

	/**
	 * @param vgl
	 *            zu vergleichendes Gangfeld
	 * @return true, wenn vgl auf gleicher Position
	 */
	protected boolean samePos(GangfeldBG vgl) {
		if (vgl.gangfeld.getRow() == gangfeld.getRow()
				&& vgl.gangfeld.getColumn() == gangfeld.getColumn())
			return true;
		else
			return false;
	}

	/**
	 * Gibt alle Appearences des Gangfeldes zurck
	 * 
	 * @param g GangfeldBG
	 * @return App-Liste
	 */
	@SuppressWarnings("unchecked")
	public Vector<Appearance> getAllAppearence(Group g) {
		Vector<Appearance> erg = new Vector<Appearance>();
		Enumeration children = g.getAllChildren();
		while(children.hasMoreElements()) {
			Object child = children.nextElement();
			if(child instanceof Primitive) {
				erg.add(((Primitive) child).getAppearance());
			} else if(child instanceof Group) {
				erg.addAll(this.getAllAppearence((Group) child));
			}
		}
		return erg;
	}

	/**
	 * Prft Zugehrigkeit von GangfeldBG und Gangfeld
	 * 
	 * @param o
	 *            zu prfendes Gangfeld
	 * @return Gleichheit der Gangfelder
	 */
	@Override
	public boolean equals(Object o) {
		if (o instanceof Gangfeld) {
			Gangfeld g = (Gangfeld) o;
			if (g.equals(gangfeld))
				return true;
			else
				return false;
		} else
			return super.equals(o);
	}
	
	/**
	 * Setzt Shining
	 * @param b
	 */
	protected void setShining(boolean b){
		if(visibleGangfeld);
		app.setTexture(null);
	}

}