package model;

import java.util.HashMap;
import java.util.Iterator;
import java.util.Observable;
import java.util.Vector;

import exceptions.GangfeldNotFoundException;
import exceptions.NoRandfeldException;

/**
 * Verwaltung der Zustnde und Logiken des Spiels
 * 
 * @author dgrosche, cdiaspero, jzimdars
 * @version final 2010-05-07
 * 
 */
public class SpielMgr extends Observable {

	// Spielphasen
	public final static int VERSCHIEBEN_PHASE = 1;
	public final static int GEHEN_PHASE = 2;

	// Definition der statischen Gangfelder mit fester Ausrichtung. Werte des
	// Arrays: 'reihe', 'spalte', 'typ', 'orientierung'.
	private final static int staticFields[][] = {
			{ 1, 1, Gangfeld.KURVE, Gangfeld.WENDE },
			{ 1, 3, Gangfeld.TSTUECK, Gangfeld.ORIGINAL },
			{ 1, 5, Gangfeld.TSTUECK, Gangfeld.ORIGINAL },
			{ 1, 7, Gangfeld.KURVE, Gangfeld.RECHTS },
			{ 3, 1, Gangfeld.TSTUECK, Gangfeld.RECHTS },
			{ 3, 3, Gangfeld.TSTUECK, Gangfeld.ORIGINAL },
			{ 3, 5, Gangfeld.TSTUECK, Gangfeld.LINKS },
			{ 3, 7, Gangfeld.TSTUECK, Gangfeld.LINKS },
			{ 5, 1, Gangfeld.TSTUECK, Gangfeld.RECHTS },
			{ 5, 3, Gangfeld.TSTUECK, Gangfeld.RECHTS },
			{ 5, 5, Gangfeld.TSTUECK, Gangfeld.WENDE },
			{ 5, 7, Gangfeld.TSTUECK, Gangfeld.LINKS },
			{ 7, 1, Gangfeld.KURVE, Gangfeld.LINKS },
			{ 7, 3, Gangfeld.TSTUECK, Gangfeld.WENDE },
			{ 7, 5, Gangfeld.TSTUECK, Gangfeld.WENDE },
			{ 7, 7, Gangfeld.KURVE, Gangfeld.ORIGINAL } };

	// Eigenschaften
	private Vector<Gangfeld> felder = new Vector<Gangfeld>();
	private Vector<Schatz> treasures = new Vector<Schatz>();
	private Vector<Spieler> spieler = new Vector<Spieler>();
	private Gangfeld freiesGangfeld;
	private Gangfeld verschiebung;
	private Schatz freierSchatz;
	private boolean wayCoordsSet = false;
	private boolean animationSet = true;

	// Spielsteuerung
	private int currentPhase = VERSCHIEBEN_PHASE;
	private int currentPlayer = 0;

	private boolean movable = true;

	/**
	 * Erstellt ein neues Spiel
	 * 
	 * @param spielernamen
	 *            Vektor mit den Namen der Spieler
	 */
	public SpielMgr(Vector<String> spielernamen) {

		// Verteilen der Gangfelder
		int counter = 0;
		for (int i = 0; i < 9; i++) {
			for (int j = 0; j < 9; j++) {
				// klickbare Randfelder
				if (i == 0 || j == 0 || i == 8 || j == 8) {
					felder.add(new Gangfeld(i, j, Gangfeld.RAND));
				}
				// statische Gangfelder
				else if (i % 2 == 1 && j % 2 == 1) {
					felder
							.add(new Gangfeld(staticFields[counter][0],
									staticFields[counter][1],
									staticFields[counter][2]));
					// Rotierung des statischen Feldes der Vorgabe anpassen.
					felder.lastElement().setOrientation(
							staticFields[counter][3]);
					// Felduebergaenge aktualisieren
					felder.lastElement().setWayConnections();
					counter++;
				}
				// zufaellig verteilte Felder
				else {
					felder.add(new Gangfeld(i, j, randomType()));
					felder.lastElement().setOrientation(randomOrientation());
					felder.lastElement().setWayConnections();
				}
			}
		}

		// Das erste freie Feld zum Verschieben des Labyrinths
		Gangfeld fg = new Gangfeld(randomType());
		fg.setWayConnections();
		setFreiesGangfeld(fg);

		// Spielfiguren auf dem Spielbrett verteilen
		for (int i = 0; i < spielernamen.size(); i++) {
			if (i == 0)
				this.addSpieler(1, 1, spielernamen.elementAt(i));
			if (i == 1)
				this.addSpieler(1, 7, spielernamen.elementAt(i));
			if (i == 2)
				this.addSpieler(7, 7, spielernamen.elementAt(i));
			if (i == 3)
				this.addSpieler(7, 1, spielernamen.elementAt(i));
		}

		// Schaetze auf Spielbrett positionieren. Zuvor muessen Spielfiguren
		// positioniert worden sein.
		treasures.add(new Schatz("Schatzkiste"));
		treasures.add(new Schatz("Ring"));
		treasures.add(new Schatz("Diamant"));
		treasures.add(new Schatz("Krone"));
		treasures.add(new Schatz("Zauberstab"));
		treasures.add(new Schatz("Buch"));
		treasures.add(new Schatz("Schlssel"));
		placeTreasuresOnBoard();

		// Jedem Spieler einen zufaellig Schatz zuweisen.
		for (Spieler s : spieler) {
			s.setSchatz(getRandomTreasure());
		}

		this.printSpielzustand();
	}

	/**
	 * @author cdiaspero
	 * 
	 *         Funktion liefert einen zufaelligen Schatz zurueck.
	 * 
	 * @return Schatz zuflliger Schatz
	 */
	private Schatz getRandomTreasure() {
		int rand = (int) ((Math.random() * treasures.size()) + 1);
		return treasures.elementAt((rand - 1));
	}

	/**
	 * @author cdiaspero
	 * 
	 *         Funktion platziert Schaetze zufaellig auf dem Spielbrett. Dabei
	 *         wird ueberprueft, ob sich auf Spielfeld bereits eine Spielfigur
	 *         oder ein anderer Schatz befindet.
	 * 
	 */
	private void placeTreasuresOnBoard() {
		for (int i = 0; i < treasures.size(); i++) {
			boolean isSet = false;
			do {
				int col = (int) ((Math.random() * 7) + 1);
				int row = (int) ((Math.random() * 7) + 1);
				try {
					if ((getGangfeld(row, col).isSpielfigurSet() == false)
							&& (getGangfeld(row, col).getTreasure() == false)) {
						treasures.elementAt(i).setPosition(row, col);
						getGangfeld(row, col).setTreasureName(
								treasures.elementAt(i).getName());
						getGangfeld(row, col).setTreasure(true);
						isSet = true;
					}
				} catch (GangfeldNotFoundException e) {
					e.printStackTrace();
				}
			} while (!isSet);
		}
	}

	/**
	 * Fgt einen Spieler dem Spiel hinzu
	 * 
	 * @param row
	 *            Reihe des Spielers
	 * @param col
	 *            Spalte des Spielers
	 * @param name
	 *            Name des Spielers
	 */
	private void addSpieler(int row, int col, String name) {
		Spieler s = new Spieler(row, col, name);
		spieler.add(s);
		try {
			getGangfeld(row, col).setSpielfigur(true);
			getGangfeld(row, col).setPlayerColor(s.getFarbe());
		} catch (GangfeldNotFoundException e) {
			e.printStackTrace();
		}
	}

	/**
	 * Markiert alle Zugmglichkeiten ausgehend von einem Gangfeld
	 * 
	 * @param x
	 *            Reihe des Ausgangsfeldes
	 * @param z
	 *            Spalte des Ausgangsfeldes
	 */
	private void setPossibleWay(int x, int z) {
		Gangfeld g = null;
		try {
			g = getGangfeld(x, z);
			g.setWaymark(true);
		} catch (GangfeldNotFoundException e) {
			e.printStackTrace();
		}
	}

	/**
	 * Findet die weiteren Zugmglichkeiten ausgehend von einem Gangfeld
	 * 
	 * @param g
	 *            Vorgngergangfeld
	 * @param color
	 *            Farbe zum einfrben
	 */
	private void findWays(Gangfeld g, String color) {
		int x = g.getRow();
		int z = g.getColumn();
		setPossibleWay(x, z);
		g.setWaymarkColor(color);

		if (g.getWays()[0]) {
			Gangfeld gang = null;
			try {
				gang = getGangfeld(x, z - 1);
			} catch (GangfeldNotFoundException e) {
				e.printStackTrace();
			}
			if (getOppositeFree(x, z - 1, 0) && !gang.getWaymark()) {
				findWays(gang, color);
			}
		}
		if (g.getWays()[1]) {
			Gangfeld gang = null;
			try {
				gang = getGangfeld(x + 1, z);
			} catch (GangfeldNotFoundException e) {
				e.printStackTrace();
			}
			if (getOppositeFree(x + 1, z, 1) && !gang.getWaymark()) {
				findWays(gang, color);
			}
		}
		if (g.getWays()[2]) {
			Gangfeld gang = null;
			try {
				gang = getGangfeld(x, z + 1);
			} catch (GangfeldNotFoundException e) {
				e.printStackTrace();
			}
			if (getOppositeFree(x, z + 1, 2) && !gang.getWaymark()) {
				findWays(gang, color);
			}
		}
		if (g.getWays()[3]) {
			Gangfeld gang = null;
			try {
				gang = getGangfeld(x - 1, z);
			} catch (GangfeldNotFoundException e) {
				e.printStackTrace();
			}
			if (getOppositeFree(x - 1, z, 3) && !gang.getWaymark()) {
				findWays(gang, color);
			}
		}
	}

	/**
	 * Gibt zurck, ob das gg. liegende Feld frei ist
	 * 
	 * @param x
	 *            Reihe des jetzigen Feldes
	 * @param z
	 *            Spalte des jetzigen Feldes
	 * @param site
	 *            Seite des jetzigen Feldes (Nord, Ost, Sd, West)
	 * @return ffnung des gg. liegenden Feldes (z.B. Sd, wenn site Nord)
	 */
	private boolean getOppositeFree(int x, int z, int site) {
		if (site == 0) {
			Gangfeld g = null;
			try {
				g = getGangfeld(x, z);
			} catch (GangfeldNotFoundException e) {
				e.printStackTrace();
			}
			return g.getWays()[2];
		}
		if (site == 1) {
			Gangfeld g = null;
			try {
				g = getGangfeld(x, z);
			} catch (GangfeldNotFoundException e) {
				e.printStackTrace();
			}
			return g.getWays()[3];
		}
		if (site == 2) {
			Gangfeld g = null;
			try {
				g = getGangfeld(x, z);
			} catch (GangfeldNotFoundException e) {
				e.printStackTrace();
			}
			return g.getWays()[0];
		}
		if (site == 3) {
			Gangfeld g = null;
			try {
				g = getGangfeld(x, z);
			} catch (GangfeldNotFoundException e) {
				e.printStackTrace();
			}
			return g.getWays()[1];
		}
		return false;
	}

	/**
	 * @return Zufallszahl zwischen 0 und 2 fr den Typ
	 */
	private int randomType() {
		int i = (int) (Math.random() * 3);
		return i;
	}

	/**
	 * @return Zufallszahl zwischen 0 und 4 fr die Orientierung
	 */
	private int randomOrientation() {
		return (int) (Math.random() * 4);
	}

	/**
	 * Setzt den nchsten Spieler dran
	 */
	private void nextPlayer() {
		if (currentPlayer == (spieler.size() - 1)) {
			currentPlayer = 0;
		} else {
			currentPlayer++;
		}
		this.setChanged();
	}

	/**
	 * Verschiebt das freie Gangfeld in das Spielfeld
	 * 
	 * @param position
	 *            Randfeld an welchem eingeschoben wird
	 * @throws NoRandfeldException
	 *             wenn position kein Randfeld
	 */
	private void verschieben(Gangfeld position) throws NoRandfeldException {
		// Ersetzen des ausgewhlten Randfeldes durch das freie Gangfeld
		position.setHighlighted(false);
		freiesGangfeld.setRow(position.getRow());
		freiesGangfeld.setColumn(position.getColumn());
		if (freierSchatz != null)
			freierSchatz.setPosition(position.getRow(), position.getColumn());
		this.setFreierSchatz(null);
		felder.add(freiesGangfeld);
		felder.remove(position);

		// Verschieben der ausgewhlten Reihe oder Spalte
		Vector<Gangfeld> toSwitch = new Vector<Gangfeld>();
		if (position.getRow() == 0) {
			for (Gangfeld g : felder) {
				if (g.getColumn() == position.getColumn()) {
					g.setRow(g.getRow() + 1);
					if (g.getRow() > 7) {
						toSwitch.add(g);
					}
				}
			}
			for (Spieler s : spieler) {
				if (s.getPosition()[1] == position.getColumn()) {
					s.setPosition(s.getPosition()[0] + 1, s.getPosition()[1]);
					if (s.getPosition()[0] > 7)
						s.setPosition(1, s.getPosition()[1]);
				}
			}
			for (Schatz t : treasures) {
				if (t.getPosition()[1] == position.getColumn()) {
					t.setPosition(t.getPosition()[0] + 1, t.getPosition()[1]);
					if (t.getPosition()[0] > 7)
						this.setFreierSchatz(t);
				}
			}
		} else if (position.getRow() == 8) {
			for (Gangfeld g : felder) {
				if (g.getColumn() == position.getColumn()) {
					g.setRow(g.getRow() - 1);
					if (g.getRow() < 1)
						toSwitch.add(g);
				}
			}
			for (Spieler s : spieler) {
				if (s.getPosition()[1] == position.getColumn()) {
					s.setPosition(s.getPosition()[0] - 1, s.getPosition()[1]);
					if (s.getPosition()[0] < 1)
						s.setPosition(7, s.getPosition()[1]);
				}
			}
			for (Schatz t : treasures) {
				if (t.getPosition()[1] == position.getColumn()) {
					t.setPosition(t.getPosition()[0] - 1, t.getPosition()[1]);
					if (t.getPosition()[0] < 1)
						this.setFreierSchatz(t);
				}
			}
		} else if (position.getColumn() == 0) {
			for (Gangfeld g : felder) {
				if (g.getRow() == position.getRow()) {
					g.setColumn(g.getColumn() + 1);
					if (g.getColumn() > 7)
						toSwitch.add(g);
				}
			}
			for (Spieler s : spieler) {
				if (s.getPosition()[0] == position.getRow()) {
					s.setPosition(s.getPosition()[0], s.getPosition()[1] + 1);
					if (s.getPosition()[1] > 7)
						s.setPosition(s.getPosition()[0], 1);
				}
			}
			for (Schatz t : treasures) {
				if (t.getPosition()[0] == position.getRow()) {
					t.setPosition(t.getPosition()[0], t.getPosition()[1] + 1);
					if (t.getPosition()[1] > 7)
						this.setFreierSchatz(t);
				}
			}
		} else if (position.getColumn() == 8) {
			for (Gangfeld g : felder) {
				if (g.getRow() == position.getRow()) {
					g.setColumn(g.getColumn() - 1);
					if (g.getColumn() < 1)
						toSwitch.add(g);
				}
			}
			for (Spieler s : spieler) {
				if (s.getPosition()[0] == position.getRow()) {
					s.setPosition(s.getPosition()[0], s.getPosition()[1] - 1);
					if (s.getPosition()[1] < 1)
						s.setPosition(s.getPosition()[0], 7);
				}
			}
			for (Schatz t : treasures) {
				if (t.getPosition()[0] == position.getRow()) {
					t.setPosition(t.getPosition()[0], t.getPosition()[1] - 1);
					if (t.getPosition()[1] < 1)
						this.setFreierSchatz(t);
				}
			}
		}

		// Vertauschen der Gangfelder und setzen des freienGangfeldes
		for (Gangfeld g : toSwitch) {
			if (g.getRow() == 0) {
				this.setFreiesGangfeld(g);
				felder.remove(g);
			} else if (g.getRow() == 8) {
				this.setFreiesGangfeld(g);
				felder.remove(g);
			} else if (g.getColumn() == 0) {
				this.setFreiesGangfeld(g);
				felder.remove(g);
			} else if (g.getColumn() == 8) {
				this.setFreiesGangfeld(g);
				felder.remove(g);
			} else if (g.getRow() == -1)
				g.setRow(0);
			else if (g.getRow() == 9)
				g.setRow(8);
			else if (g.getColumn() == -1)
				g.setColumn(0);
			else if (g.getColumn() == 9)
				g.setColumn(8);
		}
		felder.add(position);
	}

	/**
	 * Lscht alle vorhandenen Werte aus den Gangfeldern
	 */
	private void deleteWaymarks() {
		for (Gangfeld g : felder) {
			g.setWaymark(false);
			g.setSpielfigur(false);
			g.setWaymarkColor(null);
		}
		this.updatePlayerPosition();
	}

	/**
	 * Aktualisiert die Spielerpositionen unter den Gangfeldern
	 */
	private void updatePlayerPosition() {
		for (Spieler s : spieler) {
			int[] pos = s.getPosition();
			try {
				Gangfeld g = this.getGangfeld(pos[0], pos[1]);
				g.setSpielfigur(true);
				g.setWaymarkColor(s.getFarbe());
			} catch (GangfeldNotFoundException e) {
				e.printStackTrace();
			}
		}

	}

	/**
	 * @author cdiaspero
	 * @return Function returns the current phase of game.
	 */
	public int getCurrentPhase() {
		return this.currentPhase;
	}

	/**
	 * @author cdiaspero
	 * @return Function returns the current player of phase of the game as
	 *         'Spieler' object.
	 */
	public Spieler getCurrentPlayer() {
		return spieler.get(currentPlayer);
	}

	/**
	 * Gibt das Gangfeld der gewnschten Position zurck
	 * 
	 * @param row
	 *            Nummer der Reihe
	 * @param col
	 *            Nummer der Splate
	 * @return Gangfeld der Position
	 * @throws GangfeldNotFoundException
	 *             wenn kein Gangfeld auf der Position
	 */
	public Gangfeld getGangfeld(int row, int col)
			throws GangfeldNotFoundException {
		Iterator<Gangfeld> iter = felder.iterator();
		while (iter.hasNext()) {
			Gangfeld erg = iter.next();
			if (erg.getRow() == row && erg.getColumn() == col)
				return erg;
		}
		throw new GangfeldNotFoundException();
	}

	/**
	 * @return alle Gangfelder als Vektor
	 */
	public Vector<Gangfeld> getAllGangfelder() {
		return this.felder;
	}

	/**
	 * @return aktuelles freies Gangfeld zum Verschieben
	 */
	public Gangfeld getFreiesGangfeld() {
		return freiesGangfeld;
	}

	/**
	 * @return alle Schtze als Vektor
	 */
	public Vector<Schatz> getSchatzList() {
		return treasures;
	}

	/**
	 * @return alle Spieler als Vektor
	 */
	public Vector<Spieler> getAllSpieler() {
		return spieler;
	}

	/**
	 * @return Randfeld zur Verschiebung des Spielbretts
	 */
	public Gangfeld getVerschiebung() {
		return this.verschiebung;
	}

	/**
	 * @return the freierSchatz
	 */
	public Schatz getFreierSchatz() {
		return freierSchatz;
	}

	/**
	 * @param freiesGangfeld
	 *            das neue freie Gangfeld
	 */
	public void setFreiesGangfeld(Gangfeld freiesGangfeld) {
		this.freiesGangfeld = freiesGangfeld;
	}

	/**
	 * @param verschiebung
	 *            Randfeld an welchem geschoben wird
	 */
	public void setVerschiebung(Gangfeld verschiebung) {
		this.verschiebung = verschiebung;
	}

	/**
	 * @param freierSchatz
	 *            the freierSchatz to set
	 */
	public void setFreierSchatz(Schatz freierSchatz) {
		this.freierSchatz = freierSchatz;
	}

	/**
	 * Setzt eine Beziehung zwischen Gangfeldern (Teil der Routine zum
	 * shortestWayPath)
	 * 
	 * @param child
	 *            Kind-Gangfeld
	 * @param parent
	 *            Eltern-Gangfeld
	 */
	public void relationship(Gangfeld child, Gangfeld parent) {
		child.setParent(parent);
	}

	/**
	 * Markiert alle mglichen Wege
	 */
	public void markingPossibleWays() {
		Spieler s = this.getCurrentPlayer();
		try {
			Gangfeld g = this.getGangfeld(s.getPosition()[0],
					s.getPosition()[1]);
			this.findWays(g, s.getFarbe());
		} catch (GangfeldNotFoundException e) {
			e.printStackTrace();
		}
	}

	/**
	 * Markiert ein Randfeld
	 * 
	 * @param g
	 *            zu markierendes Feld
	 * @param value
	 *            markieren oder demarkieren?
	 * @throws NoRandfeldException
	 *             wenn Feld kein Randfeld
	 */
	public void highlightingRandfeld(Gangfeld g, boolean value)
			throws NoRandfeldException {
		g.setHighlighted(value);
		this.setChanged();
	}

	/**
	 * Verschiebt einen Spieler
	 * 
	 * @param x
	 *            Reihe des Spielers
	 * @param z
	 *            Spalte des Spielers
	 */
	public void movePlayer(int x, int z) {
		this.getCurrentPlayer().setPosition(x, z);
		this.getCurrentPlayer().setVisible(false);
		for (Gangfeld g : felder) {
			if (g.getRow() == x && g.getColumn() == z) {
				if (g.getTreasureName().equals(
						getCurrentPlayer().getSchatz().getName())) {
					System.out.println("Schatz gefunden.");
					// neuen Schatz zuweisen
					getCurrentPlayer().setSchatz(getRandomTreasure());
					System.out.println("Naechster Schatz: "
							+ getCurrentPlayer().getSchatz().getName());
				}
			}
		}
		this.setChanged();
	}

	/**
	 * Verschiebt das freie Gangfeld an der Verschieben-Position
	 */
	public HashMap<String, Gangfeld> verschieben() {
		HashMap<String, Gangfeld> erg = new HashMap<String, Gangfeld>();
		erg.put("Position", verschiebung);
		erg.put("Einschub", freiesGangfeld);
		try {
			this.verschieben(verschiebung);
		} catch (NoRandfeldException e) {
			e.printStackTrace();
		}
		this.markingPossibleWays();
		this.setChanged();
		return erg;
	}

	/**
	 * Rotiert ein Gangfeld um 90 mathematisch positiv
	 * 
	 * @param g
	 *            zu rotierendes Gangfeld
	 */
	public void rotate(Gangfeld g) {
		int rot = (g.getOrientation() + 1);
		g.setOrientation(rot % 4);
		this.setChanged();
	}

	/**
	 * Prft ob ein Schatz erreicht wurde
	 */
	private void checkTreasures() {
		Spieler aktSpieler = this.getCurrentPlayer();
		for (Schatz s : treasures) {
			if (aktSpieler.getSchatz().equals(s)) {
//				System.out.println("HURRA " + s.getName() + " gefunden!");
				// neuen Schatz zuweisen
//				aktSpieler.setSchatz(getRandomTreasure());
//				System.out.println("Naechster Schatz: "
//						+ aktSpieler.getSchatz().getName());
			}
		}
	}

	/**
	 * Nchste Spielphase
	 */
	public void weiter() {
		switch (currentPhase) {
		case VERSCHIEBEN_PHASE: {
			System.out.println("verschieben phase ist beendet");
			currentPhase = 2;
		}
			break;
		case GEHEN_PHASE: {
			System.out.println("gehen phase ist beendet");
			this.checkTreasures();
			currentPhase = 1;
			this.deleteWaymarks();
			nextPlayer();
		}
			break;
		}
		this.setChanged();
		System.out.println(printSpielzustand());
	}

	/**
	 * Druckt Spielzustand aus
	 */
	public String printSpielzustand() {
		String phase = "Verschieben";
		if (getCurrentPhase() == SpielMgr.GEHEN_PHASE) {
			phase = "Gehen";
		}
		 return "Phase: " + phase + ", Spieler: "
				+ getCurrentPlayer().getName() + ", Position: "
				+ getCurrentPlayer().getPosition()[0] + "/"
				+ getCurrentPlayer().getPosition()[1];
	}

	/**
	 * @param wayCoordsSet
	 *            the wayCoordsSet to set
	 */
	public void setWayCoords(boolean wayCoordsSet) {
		this.setChanged();
		this.wayCoordsSet = wayCoordsSet;
	}

	/**
	 * @return the wayCoordsSet
	 */
	public boolean isWayCoordsSet() {
		return wayCoordsSet;
	}

	/**
	 * @param b the animationSet to set
	 */
	public void setAnimation(boolean b) {
		animationSet = b;
	}

	/**
	 * @return the animationSet
	 */
	public boolean getAnimationSet() {
		return animationSet;
	}

	/**
	 * @return the movable
	 */
	public boolean getMovable() {
		return movable;
	}

	/**
	 * @param b the movable to set
	 * @param pauseTime the pause time to set
	 */
	public void setMovable(boolean b, int pauseTime) {
		this.movable = b;
		if (!b) {
			new MoveTimer(pauseTime, this, null);
		}
	}

}