package model;

import exceptions.NoRandfeldException;

/**
 * Datenobjekt Gangfeld
 * 
 * @author dgrosche, jzimdars, cdiaspero
 * @version final 2010-05-07
 */
public class Gangfeld {

	// Gangfeld-Typen
	public final static int KURVE = 0;
	public final static int GERADE = 1;
	public final static int TSTUECK = 2;
	public final static int FEST = 3;
	public final static int RAND = 4;

	// Rotationsorientierungen
	public final static int ORIGINAL = 0;
	public final static int LINKS = 1;
	public final static int WENDE = 2;
	public final static int RECHTS = 3;

	// Feldoeffnungen (false = geschlossen, true = offen)
	public final static boolean[][] oeffnungenKurve = {
			{ true, false, false, true }, { false, false, true, true },
			{ false, true, true, false }, { true, true, false, false } };
	public final static boolean[][] oeffnungenGerade = {
			{ true, false, true, false }, { false, true, false, true },
			{ true, false, true, false }, { false, true, false, true } };
	public final static boolean[][] oeffnungenTStueck = {
			{ true, true, true, false }, { true, true, false, true },
			{ true, false, true, true }, { false, true, true, true } };
	// interne Eigenschaften
	private int row, column, orientation, typ;
	private boolean highlighted;

	// Status Feldoeffnung
	private boolean north;
	private boolean east;
	private boolean south;
	private boolean west;
	private boolean spielfigurSet;
	private boolean treasureSet = false;
	private String treasureName = "";
	private String spielfigurColor = "";
	private boolean waymarkSet = false;
	private String waymarkColor = "";

	// zur Wegfindung der Spielfigur
	private Gangfeld parent = null;

	/**
	 * Erstellt ein neues Gangfeld
	 * 
	 * @param reihe
	 *            Reihe des Gangfeldes um es spter wiederfinden zu knnen
	 * @param spalte
	 *            Spalte des Gangfeldes um es spter wiederfinden zu knnen
	 * @param typ
	 *            Das ist der entscheidende Wert: 0 = Gangfeld.KURVE = Kurve von
	 *            links nach oben 1 = Gangfeld.GERADE = Gerade von unten nach
	 *            oben 2 = Gangfeld.TSTUECK = T-Stck links geschlossen 3 =
	 *            Gangfeld.FEST = Ein festes unbewegliches Feld
	 */
	public Gangfeld(int reihe, int spalte, int typ) {
		// bergeben der Werte
		this.row = reihe;
		this.column = spalte;
		this.typ = typ;
		this.orientation = Gangfeld.ORIGINAL;

		// Set open way connections to standard.
		this.setWayConnections();
	}

	/**
	 * Erstellt ein neues Gangfeld ohne Position
	 * 
	 * @param typ
	 *            Das ist der entscheidende Wert: 0 = Gangfeld.KURVE = Kurve von
	 *            links nach oben 1 = Gangfeld.GERADE = Gerade von unten nach
	 *            oben 2 = Gangfeld.TSTUECK = T-Stck links geschlossen 3 =
	 *            Gangfeld.FEST = Ein festes unbewegliches Feld
	 */
	public Gangfeld(int typ) {
		this.typ = typ;
		this.orientation = Gangfeld.LINKS;
	}

	/**
	 * @return Reihennummer
	 */
	public int getRow() {
		return row;
	}

	/**
	 * @return Spaltennummer
	 */
	public int getColumn() {
		return column;
	}

	/**
	 * @return Ganzahl zwischen 0 und 3 als Drehung von 90 Grad
	 */
	public int getOrientation() {
		return this.orientation;
	}

	/**
	 * @return Typ - KURVE, GERADE, TSTUECK, FEST, RAND
	 */
	public int getTyp() {
		return this.typ;
	}

	/**
	 * @return the highlighted
	 */
	public boolean isHighlighted() {
		return highlighted;
	}

	/**
	 * @return Returns boolean array (false = closed, true = open) with way
	 *         connections of field. Structure of array: [north, east, south,
	 *         west].
	 */
	public boolean[] getWays() {
		boolean[] ways = new boolean[4];
		ways[0] = north;
		ways[1] = east;
		ways[2] = south;
		ways[3] = west;
		return ways;
	}

	/**
	 * @return true, if soielfigur on same position
	 */
	public boolean isSpielfigurSet() {
		return spielfigurSet;
	}

	/**
	 * @return true, if treasure on same position
	 */
	public boolean getTreasure() {
		return treasureSet;
	}

	/**
	 * @return treasure name or null if not set
	 */
	public String getTreasureName() {
		return this.treasureName;
	}

	/**
	 * @return true, if figure is allowed to move here
	 */
	public boolean getWaymark() {
		return waymarkSet;
	}

	/**
	 * @return highlighting color equals figure color
	 */
	public String getWaymarkColor() {
		return waymarkColor;
	}

	/**
	 * @return color of figure, if figure on same pos
	 */
	public String getPlayerColor() {
		return this.spielfigurColor;
	}

	/**
	 * Gibt das Gangfeld zurck ber welches der 
	 * Wegsuchalgorythmus zu dem aktuellen Gangfeld
	 * gekommen ist
	 * 
	 * @param parent
	 *            Gangfeld
	 */
	
	
	public Gangfeld getParent() {
		return parent;
	}

	/**
	 * Entfernt der Verweis auf das bergeordnete Gangfeld
	 * 
	 * @param parent
	 *            Gangfeld
	 */
	
	public void removeParent() {
		parent = null;
	}

	/**
	 * Setzt die Reihe
	 * 
	 * @param reihe
	 *            Reihe
	 */
	protected void setRow(int reihe) {
		this.row = reihe;
	}

	/**
	 * Setzt die Spalte
	 * 
	 * @param spalte
	 *            Spalte
	 */
	protected void setColumn(int spalte) {
		this.column = spalte;
	}

	/**
	 * Setzt Orientierung auf Ganzzahl zwischen 0 und 3
	 * 
	 * @param orientierung
	 *            ORIGINAL, LINKS, WENDE, RECHTS
	 */
	protected void setOrientation(int orientierung) {
		this.orientation = orientierung;
		this.setWayConnections();
	}

	/**
	 * @param highlighted
	 *            the highlighted to set
	 * @throws NoRandfeldException
	 *             nur Randfelder knnen markiert werden
	 */
	protected void setHighlighted(boolean highlighted)
			throws NoRandfeldException {
		if (this.getTyp() != Gangfeld.RAND)
			throw new NoRandfeldException();
		this.highlighted = highlighted;
	}

	/**
	 * @param b
	 *            wenn Spielfigur auf gleicher Pos
	 */
	protected void setSpielfigur(boolean b) {
		spielfigurSet = b;
	}

	/**
	 * @param b
	 *            wenn setzen hierauf mglich
	 */
	protected void setWaymark(boolean b) {
		waymarkSet = b;
	}

	/**
	 * @param c
	 *            name der Farbe zum einfrben
	 */
	protected void setPlayerColor(String c) {
		this.spielfigurColor = c;
	}

	/**
	 * @param b
	 *            wenn Schatz auf gleicher Pos
	 */
	protected void setTreasure(boolean b) {
		this.treasureSet = b;
	}

	/**
	 * @param n
	 *            Name des Schatzes auf dieser Pos
	 */
	protected void setTreasureName(String n) {
		this.treasureName = n;
	}

	/**
	 * @param c
	 *            Name der Farbe zum einfrben
	 */
	protected void setWaymarkColor(String c) {
		this.waymarkColor = c;
	}

	/**
	 * Setzt das Gangfeld ber welches der 
	 * Wegsuchalgorythmus zu dem aktuellen Gangfeld
	 * gekommen ist
	 * 
	 * @param parent
	 *            Gangfeld
	 */
	protected void setParent(Gangfeld g) {
		parent = g;
	}

	/**
	 * @author cdiaspero
	 * 
	 *         Function sets way connections of field, using defined preset
	 *         arrays (oeffnungenKurve, oeffnungenGerade, oeffnungenTStueck).
	 * 
	 */
	protected void setWayConnections() {
		boolean[] wayConnections = new boolean[4];

		// Switch for type of field.
		switch (this.typ) {
		// Type of field: 'Kurve'
		case 0:
			wayConnections = oeffnungenKurve[this.getOrientation()];
			break;
		// Type of field: 'Gerade'
		case 1:
			wayConnections = oeffnungenGerade[this.getOrientation()];
			break;
		// Type of field: 'TStueck'
		case 2:
			wayConnections = oeffnungenTStueck[this.getOrientation()];
			break;
		// Type of field: 'Randfeld'
		default:
			wayConnections[0] = false;
			wayConnections[1] = false;
			wayConnections[2] = false;
			wayConnections[3] = false;
			break;
		}

		// Set way connections of current field.
		this.north = wayConnections[0];
		this.east = wayConnections[1];
		this.south = wayConnections[2];
		this.west = wayConnections[3];

	}

	/**
	 * Gibt Gangfeld samt Art und Position zurck
	 */
	@Override
	public String toString() {
		return "Gangfeld - " + typ + " - " + this.getColumn() + ";"
				+ this.getRow();
	}

	/**
	 * Prft ob zwei Gangfelder identisch sind. Ausschlaggebend sind Typ,
	 * Position und Orientierung.
	 */
	@Override
	public boolean equals(Object vergleich) {
		if (vergleich instanceof Gangfeld) {
			Gangfeld g = (Gangfeld) vergleich;
			if (this.typ == g.getTyp() && this.row == g.getRow()
					&& this.column == g.getColumn()
					&& this.orientation == g.getOrientation())
				return true;
		}
		return false;
	}

}